<h2>Why is this an issue?</h2>
<p>Constructors should not be asynchronous because they are meant to initialize an instance of a class synchronously. While you can technically run a
promise in a constructor, the instance will be returned before the asynchronous operation completes, which can lead to potential issues if the rest of
your code expects the object to be fully initialized. While returning the promise may seem the solution for this problem, it is not standard practice
and can lead to unexpected behavior. Returning a promise (or, in general, any object from another class) from a constructor can be misleading and is
considered a bad practice as this leads to unexpected results with inheritance and the <code>instanceof</code> operator.</p>

```js
class MyClass {
  constructor() {
    Promise.resolve().then(() => this.data = fetchData()); // Noncompliant, this.data will be undefined in the new instance
  }
}
```
<p>Instead, you can create an asynchronous method that performs the necessary asynchronous operations after the object has been constructed</p>

```js
class MyClass {
  constructor() {
    this.data = null;
  }

  async initialize() {
    this.data = await fetchData();
  }
}

(async () => {
  const myObject = new MyClass();
  await myObject.initialize();
})();
```
<p>Otherwise, you can use a static factory method that returns a promise resolving to an instance of the class after performing the asynchronous
operations.</p>

```js
class MyClass {
  constructor(data) {
    this.data = data;
  }

  static async create() {
    const data = await fetchData();
    return new MyClass(data);
  }
}

(async () => {
  const myObject = await MyClass.create();
})();
```
<p>Using an asynchronous factory function or an initialization method provides a more conventional, readable, and maintainable approach to handling
asynchronous initialization in JavaScript.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/new">new</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise">Promise</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/instanceof">instanceof</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Inheritance_and_the_prototype_chain">Inheritance and the
  prototype chain</a> </li>
</ul>