<p>This vulnerability makes it possible that an encrypted communication is intercepted.</p>
<h2>Why is this an issue?</h2>
<p>Transport Layer Security (TLS) provides secure communication between systems over the internet by encrypting the data sent between them. The role
of certificate validation in this process is to ensure that a system is indeed the one it claims to be, adding an extra layer of trust and
security.</p>
<p>When certificate validation is disabled, the client skips this critical check. This creates an opportunity for attackers to pose as a trusted
entity and intercept, manipulate, or steal the data being transmitted.</p>
<h3>What is the potential impact?</h3>
<p>Establishing trust in a secure way is a non-trivial task. When you disable certificate validation, you are removing a key mechanism designed to
build this trust in internet communication, opening your system up to a number of potential threats.</p>
<h4>Identity spoofing</h4>
<p>If a system does not validate certificates, it cannot confirm the identity of the other party involved in the communication. An attacker can
exploit this by creating a fake server and masquerading it as a legitimate one. For example, they might set up a server that looks like your bank’s
server, tricking your system into thinking it is communicating with the bank. This scenario, called identity spoofing, allows the attacker to collect
any data your system sends to them, potentially leading to significant data breaches.</p>
<h4>Loss of data integrity</h4>
<p>When TLS certificate validation is disabled, the integrity of the data you send and receive cannot be guaranteed. An attacker could modify the data
in transit, and you would have no way of knowing. This could range from subtle manipulations of the data you receive to the injection of malicious
code or malware into your system. The consequences of such breaches of data integrity can be severe, depending on the nature of the data and the
system.</p>
<h2>How to fix it in Java Cryptographic Extension</h2>
<h3>Code examples</h3>
<p>The following code contains examples of disabled certificate validation.</p>
<p>The certificate validation gets disabled by overriding <code>X509TrustManager</code> with an empty implementation. It is highly recommended to use
the original implementation.</p>
<h4>Noncompliant code example</h4>

```kotlin
val trustAllCerts = arrayOf<TrustManager>(object : X509TrustManager {
  @Throws(CertificateException::class)
  override fun checkClientTrusted(chain: Array<java.security.cert.X509Certificate>, authType: String) {
  } // Noncompliant

  @Throws(CertificateException::class)
  override fun checkServerTrusted(chain: Array<java.security.cert.X509Certificate>, authType: String) {
  } // Noncompliant

  override fun getAcceptedIssuers(): Array<java.security.cert.X509Certificate> {
   return arrayOf()
  }
})
```
<h3>How does this work?</h3>
<p>Addressing the vulnerability of disabled TLS certificate validation primarily involves re-enabling the default validation.</p>
<p>To avoid running into problems with invalid certificates, consider the following sections.</p>
<h4>Using trusted certificates</h4>
<p>If possible, always use a certificate issued by a well-known, trusted CA for your server. Most programming environments come with a predefined list
of trusted root CAs, and certificates issued by these authorities are validated automatically. This is the best practice, and it requires no
additional code or configuration.</p>
<h4>Working with self-signed certificates or non-standard CAs</h4>
<p>In some cases, you might need to work with a server using a self-signed certificate, or a certificate issued by a CA not included in your trusted
roots. Rather than disabling certificate validation in your code, you can add the necessary certificates to your trust store.</p>
<p>Here is a sample command to import a certificate to the Java trust store:</p>

```kotlin
keytool -import -alias myserver -file myserver.crt -keystore cacerts
```
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">OWASP Top 10 2021 Category A2</a> - Cryptographic Failures </li>
  <li> <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">OWASP Top 10 2021 Category A5</a> - Security Misconfiguration </li>
  <li> <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">OWASP Top 10 2021 Category A7</a> - Identification and
  Authentication Failures </li>
  <li> <a href="https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">OWASP Top 10 2017 Category A3</a> - Sensitive Data
  Exposure </li>
  <li> <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">OWASP Top 10 2017 Category A6</a> - Security
  Misconfiguration </li>
  <li> <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m3-insecure-communication">OWASP Mobile Top 10 2016 Category M3</a> - Insecure
  Communication </li>
  <li> <a href="https://mobile-security.gitbook.io/masvs/security-requirements/0x10-v5-network_communication_requirements">Mobile AppSec Verification
  Standard</a> - Network Communication Requirements </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/295">MITRE, CWE-295</a> - Improper Certificate Validation </li>
  <li> <a
  href="https://wiki.sei.cmu.edu/confluence/display/java/MSC61-J.+Do+not+use+insecure+or+weak+cryptographic+algorithms">https://wiki.sei.cmu.edu/confluence/display/java/MSC61-J.+Do+not+use+insecure+or+weak+cryptographic+algorithms</a> </li>
</ul>