<p>This vulnerability makes it possible that the cleartext of the encrypted message might be recoverable without prior knowledge of the key.</p>
<h2>Why is this an issue?</h2>
<p>Encryption algorithms are essential for protecting sensitive information and ensuring secure communication in various domains. They are used for
several important reasons:</p>
<ul>
  <li> Confidentiality, privacy, and intellectual property protection. </li>
  <li> Security during transmission or on storage devices. </li>
  <li> Data integrity, general trust, and authentication. </li>
</ul>
<p>When selecting encryption algorithms, tools, or combinations, you should also consider two things:</p>
<ol>
  <li> No encryption is unbreakable. </li>
  <li> The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame. </li>
</ol>
<p>For these reasons, as soon as cryptography is included in a project, it is important to choose encryption algorithms that are considered strong and
secure by the cryptography community.</p>
<h3>What is the potential impact?</h3>
<p>The cleartext of an encrypted message might be recoverable. Additionally, it might be possible to modify the cleartext of an encrypted message.</p>
<p>Below are some real-world scenarios that illustrate some impacts of an attacker exploiting the vulnerability.</p>
<h4>Theft of sensitive data</h4>
<p>The encrypted message might contain data that is considered sensitive and should not be known to third parties.</p>
<p>By using a weak algorithm the likelihood that an attacker might be able to recover the cleartext drastically increases.</p>
<h4>Additional attack surface</h4>
<p>By modifying the cleartext of the encrypted message it might be possible for an attacker to trigger other vulnerabilities in the code. Encrypted
values are often considered trusted, since under normal circumstances it would not be possible for a third party to modify them.</p>
<h2>How to fix it in Java Cryptographic Extension</h2>
<h3>Code examples</h3>
<p>The following code contains examples of algorithms that are not considered highly resistant to cryptanalysis and thus should be avoided.</p>
<h4>Noncompliant code example</h4>

```kotlin
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException
import javax.crypto.Cipher

fun main(args: Array<String>) {
    try {
        val des = Cipher.getInstance("DES") // Noncompliant
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<h4>Compliant solution</h4>

```kotlin
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException
import javax.crypto.Cipher

fun main(args: Array<String>) {
    try {
        val aes = Cipher.getInstance("AES/GCM/NoPadding")
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<h3>How does this work?</h3>
<h4>Use a secure algorithm</h4>
<p>It is highly recommended to use an algorithm that is currently considered secure by the cryptographic community. A common choice for such an
algorithm is the Advanced Encryption Standard (AES).</p>
<p>For block ciphers, it is not recommended to use algorithms with a block size that is smaller than 128 bits.</p>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">OWASP Top 10 2021 Category A2</a> - Cryptographic Failures </li>
  <li> <a href="https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">OWASP Top 10 2017 Category A3</a> - Sensitive Data
  Exposure </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/327">MITRE, CWE-327</a> - Use of a Broken or Risky Cryptographic Algorithm </li>
  <li> <a href="https://mobile-security.gitbook.io/masvs/security-requirements/0x08-v3-cryptography_verification_requirements">Mobile AppSec
  Verification Standard</a> - Cryptography Requirements </li>
  <li> <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography">OWASP Mobile Top 10 2016 Category M5</a> -
  Insufficient Cryptography </li>
</ul>