<h2>Why is this an issue?</h2>
<p>Some functions return <code>kotlinx.coroutines.Deferred</code> to eventually communicate the result of an asynchronous operation. This is
necessary, as callers do not wait for the result of an operation when it is launched asynchronously. Even if the result may be relevant later on,
other tasks can be completed while waiting for the asynchronous operation’s result to become available. Hence, to avoid blocking the caller, functions
can trigger a task to be run and then immediately return a <code>Deferred</code> instance, which will contain the result once the background task is
complete.</p>
<p><code>Deferred</code> (aka Future, Promise, etc) provides an <code>await()</code> function, which suspends the caller coroutine until the
asynchronous task is complete and returns the result of the execution. By not using the <code>Deferred</code> return value, the result of the
corresponding asynchronously launched task is lost. This could point to an issue in the code, where data is not passed along as intended.</p>
<p>For instance, the Kotlin coroutines API provides both the functions <code>async</code> and <code>launch</code> as ways to launch asynchronous work.
The key difference here is their return type. <code>launch</code> starts a new coroutine without blocking the current thread and returns a reference
to the coroutine as a Job. This function is not designed to return a result, i.e. follows the idea of "fire and forget". <code>async</code> creates a
coroutine and returns its future result as an implementation of <code>Deferred</code>.</p>
<p>Ask yourself whether:</p>
<ul>
  <li> You really need whatever result is calculated by the asynchronous operation. If not, you may be better off using a function that does not
  return <code>Deferred</code>. </li>
  <li> You should be using the <code>Deferred</code> return value and fetching some data from it later on, for instance by calling
  <code>await()</code> on it. </li>
</ul>
<p>This rule raises an issue when a function returning the type <code>kotlinx.coroutines.Deferred</code> is used without the result of the operation
being retrieved.</p>
<h3>Noncompliant code example</h3>
<p>Here <code>coroutineScope</code> returns the <code>Deferred</code> instance initially returned by <code>async</code>. It is not used in any way
afterwards, which could point to an issue in the business logic.</p>

```kotlin
suspend fun doSomething() {
    coroutineScope { // Noncompliant
        async {
            // Do some work
            "result"
        }
    }
}
```
<h3>Compliant solution</h3>
<p>Using <code>launch</code>:</p>

```kotlin
suspend fun doSomething() {
    coroutineScope {
        launch {
            // Do some work
        }
    }
}
```
<p>Using <code>await</code> to retrieve the deferred result of the asynchronous operation:</p>

```kotlin
suspend fun doSomething(): String {
    return coroutineScope {
        val asyncTask = async {
            // Do some work
            "result"
        }
        // Return the result, possibly with some other processing before, by calling await() on the Deferred instance
        asyncTask.await()
    }
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://kotlinlang.org/docs/composing-suspending-functions.html#concurrent-using-async">Concurrent using async</a> </li>
</ul>