<h2>Why is this an issue?</h2>
<p>In JavaScript, <code>NaN</code> stands for "Not-a-Number." It is a special value representing a numeric data type that is not a valid number.
<code>NaN</code> is returned as a result when an arithmetic operation or mathematical function is performed, and the result is undefined or
unrepresentable as a valid number.</p>
<p>Comparing a value with <code>NaN</code> in JavaScript can be problematic because of the way <code>NaN</code> behaves in comparison operations. The
reason is that <code>NaN</code> is not equal to any value, including itself, and this behavior can lead to unexpected results.</p>

```ts
const a = NaN;

if (a === NaN) { // Noncompliant: Always false
  console.log("a is not a number"); // This is dead code
}

if (a !== NaN) { // Noncompliant: Always true
  console.log("a is not NaN"); // This statement is not necessarily true
}
```
<p>To check if a value is <code>NaN</code>, you should use the <code>isNaN()</code> function:</p>

```ts
const a = NaN;

if (isNaN(a)) {
  console.log("a is not a number");
}

if (!isNaN(a)) {
  console.log("a is not NaN");
}
```
<p>Keep in mind that <code>isNaN()</code> can be a bit quirky since it tries to convert its argument into a number before checking if it is
<code>NaN</code>. If the argument cannot be converted into a number, <code>isNaN()</code> will return true, which may not be the desired behavior in
all cases.</p>
<p>Instead, you should prefer using the <code>Number.isNaN()</code> method over <code>isNaN()</code> to perform a strict check for <code>NaN</code>
without any type conversion:</p>

```ts
const a = NaN;

if (Number.isNaN(a)) {
  console.log("a is not a number");
}

if (!Number.isNaN(a)) {
  console.log("a is not NaN");
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/NaN"><code>NaN</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/isNaN"><code>isNaN()</code></a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/isNaN"><code>Number.isNaN()</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Glossary/Type_Conversion">Type conversion</a> </li>
</ul>