<p>Cross-origin communication allows different websites to interact with each other. This interaction is typically achieved through mechanisms like
AJAX requests, WebSockets, or postMessage API. However, a vulnerability can arise when these communications are not properly secured by verifying
their origins.</p>
<h2>Why is this an issue?</h2>
<p>Without origin verification, the target website cannot distinguish between legitimate requests from its own pages and malicious requests from an
attacker’s site. The attacker can craft a malicious website or script that sends requests to a target website where the user is already
authenticated.</p>
<p>This vulnerability class is not about a single specific user input or action, but rather a series of actions that lead to an insecure cross-origin
communication.</p>
<h3>What is the potential impact?</h3>
<p>The absence of origin verification during cross-origin communications can lead to serious security issues.</p>
<h4>Data Breach</h4>
<p>If an attacker can successfully exploit this vulnerability, they may gain unauthorized access to sensitive data. For instance, a user’s personal
information, financial details, or other confidential data could be exposed. This not only compromises the user’s privacy but can also lead to
identity theft or financial loss.</p>
<h4>Unauthorized Actions</h4>
<p>An attacker could manipulate the communication between websites to perform actions on behalf of the user without their knowledge. This could range
from making unauthorized purchases to changing user settings or even deleting accounts.</p>
<h2>How to fix it</h2>
<p>When sending a message, avoid using <code>*</code> for the target origin (it means no preference). Instead define it explicitly so the message will
only be dispatched to this URI. When receiving the message, verify the orgin to be sure that it is sent by an authorized sender.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>When sending a message:</p>

```ts
var iframe = document.getElementById("testiframe");
iframe.contentWindow.postMessage("hello", "*"); // Noncompliant: * is used
```
<p>When receiving a message:</p>

```ts
window.addEventListener("message", function(event) { // Noncompliant: no checks are done on the origin property.
  console.log(event.data);
 });
```
<h4>Compliant solution</h4>
<p>When sending a message:</p>

```ts
var iframe = document.getElementById("testiframe");
iframe.contentWindow.postMessage("hello", "https://secure.example.com");
```
<p>When receiving a message:</p>

```ts
window.addEventListener("message", function(event) {
  if (event.origin !== "http://example.org")
    return;

  console.log(event.data)
});
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/API/Window/postMessage">postMessage API</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A01_2021-Broken_Access_Control/">Top 10 2021 Category A1 - Broken Access Control</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control">Top 10 2017 Category A5 - Broken Access Control</a>
  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/345">CWE-345 - Insufficient Verification of Data Authenticity</a> </li>
</ul>