<p>This rule raises an issue when the code cognitive complexity of a function is above a certain threshold.</p>
<h2>Why is this an issue?</h2>
<p>Cognitive Complexity is a measure of how hard it is to understand the control flow of a unit of code. Code with high cognitive complexity is hard
to read, understand, test, and modify.</p>
<p>As a rule of thumb, high cognitive complexity is a sign that the code should be refactored into smaller, easier-to-manage pieces.</p>
<h3>Which syntax in code does impact cognitive complexity score?</h3>
<p>Here are the core concepts:</p>
<ul>
  <li> <strong>Cognitive complexity is incremented each time the code breaks the normal linear reading flow.</strong><br> This concerns, for example,
  loop structures, conditionals, catches, switches, jumps to labels, and conditions mixing multiple operators. </li>
  <li> <strong>Each nesting level increases complexity.</strong><br> During code reading, the deeper you go through nested layers, the harder it
  becomes to keep the context in mind. </li>
  <li> <strong>Method calls are free</strong><br> A well-picked method name is a summary of multiple lines of code. A reader can first explore a
  high-level view of what the code is performing then go deeper and deeper by looking at called functions content.<br> <em>Note:</em> This does not
  apply to recursive calls, those will increment cognitive score. </li>
</ul>
<p>The method of computation is fully detailed in the pdf linked in the resources.</p>
<p>Note that the calculation of cognitive complexity at function level deviates from the documented process. Given the functional nature of
JavaScript, nesting functions is a prevalent practice, especially within frameworks like React.js. Consequently, the cognitive complexity of functions
remains independent from one another. This means that the complexity of a nesting function does not increase with the complexity of nested
functions.</p>
<h3>What is the potential impact?</h3>
<p>Developers spend more time reading and understanding code than writing it. High cognitive complexity slows down changes and increases the cost of
maintenance.</p>
<h3>Exceptions</h3>
<p>Cognitive complexity calculations exclude logical expressions using the <code>||</code> and <code>??</code> operators when applied within the
context of default value code patterns.</p>

```ts
function greet(name) {
    name = name || 'Guest';
    console.log('Hello, ' + name + '!');
}
```
<h2>How to fix it</h2>
<p>Reducing cognitive complexity can be challenging.<br> Here are a few suggestions:</p>
<ul>
  <li> <strong>Extract complex conditions in a new function.</strong><br> Mixed operators in condition will increase complexity. Extracting the
  condition in a new function with an appropriate name will reduce cognitive load. </li>
  <li> <strong>Break down large functions.</strong><br> Large functions can be hard to understand and maintain. If a function is doing too many
  things, consider breaking it down into smaller, more manageable functions. Each function should have a single responsibility. </li>
  <li> <strong>Avoid deep nesting by returning early.</strong><br> To avoid the nesting of conditions, process exceptional cases first and return
  early. </li>
  <li> <strong>Use null-safe operations (if available in the language).</strong><br> When available the <code>.?</code> or <code>??</code> operator
  replaces multiple tests and simplifies the flow. </li>
</ul>
<h3>Code examples</h3>
<p><strong>Extraction of a complex condition in a new function.</strong></p>
<h4>Noncompliant code example</h4>
<p>The code is using a complex condition and has a cognitive cost of 3.</p>

```ts
function calculateFinalPrice(user, cart) {
  let total = calculateTotal(cart);
  if (user.hasMembership                       // +1 (if)
    && user.orders > 10                        // +1 (more than one condition)
    && user.accountActive
    && !user.hasDiscount
    || user.orders === 1) {                    // +1 (change of operator in condition)
      total = applyDiscount(user, total);
  }
  return total;
}
```
<h4>Compliant solution</h4>
<p>Even if the cognitive complexity of the whole program did not change, it is easier for a reader to understand the code of the
<code>calculateFinalPrice</code> function, which now only has a cognitive cost of 1.</p>

```ts
function calculateFinalPrice(user, cart) {
  let total = calculateTotal(cart);
  if (isEligibleForDiscount(user)) {       // +1 (if)
    total = applyDiscount(user, total);
  }
  return total;
}

function isEligibleForDiscount(user) {
  return user.hasMembership
    && user.orders > 10                     // +1 (more than one condition)
    && user.accountActive
    && !user.hasDiscount
    || user.orders === 1                    // +1 (change of operator in condition)
}
```
<p><strong>Break down large functions.</strong></p>
<h4>Noncompliant code example</h4>
<p>For example, consider a function that calculates the total price of a shopping cart, including sales tax and shipping.<br> <em>Note:</em> The code
is simplified here, to illustrate the purpose. Please imagine there is more happening in the <code>for</code> loops.</p>

```ts
function calculateTotal(cart) {
  let total = 0;
  for (let i = 0; i < cart.length; i++) {       // +1 (for)
    total += cart[i].price;
  }

  // calculateSalesTax
  for (let i = 0; i < cart.length; i++) {       // +1 (for)
    total += 0.2 * cart[i].price;
  }

  //calculateShipping
  total += 5 * cart.length;

  return total;
}
```
<p>This function could be refactored into smaller functions: The complexity is spread over multiple functions and the complex
<code>calculateTotal</code> has now a complexity score of zero.</p>
<h4>Compliant solution</h4>

```ts
function calculateTotal(cart) {
  let total = calculateSubtotal(cart);
  total += calculateSalesTax(cart);
  total += calculateShipping(cart);
  return total;
}

function calculateSubtotal(cart) {
  let subTotal = 0;
  for (const item of cart) {        // +1 (for)
    subTotal += item.price;
  }
  return subTotal;
}

function calculateSalesTax(cart) {
  let salesTax = 0;
  for (const item of cart) {        // +1 (for)
    salesTax += 0.2 * item.price;
  }
  return salesTax;
}

function calculateShipping(cart) {
  return 5 * cart.length;
}
```
<p><strong>Avoid deep nesting by returning early.</strong></p>
<h4>Noncompliant code example</h4>
<p>The below code has a cognitive complexity of 6.</p>

```ts
function calculateDiscount(price, user) {
  if (isEligibleForDiscount(user)) {  // +1 ( if )
    if (user?.hasMembership) {        // +2 ( nested if )
      return price * 0.9;
  } else if (user?.orders === 1 ) {   // +1 ( else )
          return price * 0.95;
    } else {                          // +1 ( else )
      return price;
    }
  } else {                            // +1 ( else )
    return price;
  }
}
```
<h4>Compliant solution</h4>
<p>Checking for the edge case first flattens the <code>if</code> statements and reduces the cognitive complexity to 3.</p>

```ts
function calculateDiscount(price, user) {
    if (!isEligibleForDiscount(user)) {  // +1 ( if )
      return price;
    }
    if (user?.hasMembership) {           // +1 ( if )
      return price * 0.9;
    }
    if (user?.orders === 1) {            // +1 ( if )
      return price * 0.95;
    }
    return price;
}
```
<p><strong>Use the optional chaining operator to access data.</strong></p>
<p>In the below code, the cognitive complexity is increased due to the multiple checks required to access the manufacturer’s name. This can be
simplified using the optional chaining operator.</p>
<h4>Noncompliant code example</h4>

```ts
let manufacturerName = null;

if (product && product.details && product.details.manufacturer) { // +1 (if) +1 (multiple condition)
    manufacturerName = product.details.manufacturer.name;
}
if (manufacturerName) { // +1 (if)
  console.log(manufacturerName);
} else {
  console.log('Manufacturer name not found');
}
```
<h4>Compliant solution</h4>
<p>The optional chaining operator will return <code>undefined</code> if any reference in the chain is <code>undefined</code> or <code>null</code>,
avoiding multiple checks:</p>

```ts
let manufacturerName = product?.details?.manufacturer?.name;

if (manufacturerName) { // +1 (if)
  console.log(manufacturerName);
} else {
  console.log('Manufacturer name not found');
}
```
<h3>Pitfalls</h3>
<p>As this code is complex, ensure that you have unit tests that cover the code before refactoring.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Sonar - <a href="https://www.sonarsource.com/docs/CognitiveComplexity.pdf">Cognitive Complexity</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> Sonar Blog - <a href="https://www.sonarsource.com/blog/5-clean-code-tips-for-reducing-cognitive-complexity/">5 Clean Code Tips for Reducing
  Cognitive Complexity</a> </li>
</ul>