<h2>Why is this an issue?</h2>
<p>Comparing the value to itself may be either a refactoring error or an outdated way of checking if the value is of a numeric data type but not a
valid number. This special numeric value is represented by <code>NaN</code> global property, where <code>NaN</code> stands for "Not-a-Number".
<code>NaN</code> is returned as a result when an arithmetic operation or mathematical function is performed, and the result is undefined or
unrepresentable as a valid number.</p>
<p>Detecting whether a value is <code>NaN</code> in JavaScript was previously problematic because of the way <code>NaN</code> behaves in comparison
operations. <code>NaN</code> is not equal to any value, including itself, so comparing the value to <code>NaN</code> will always return false. In
other words, if a value is not equal to itself, it can only be <code>NaN</code>.</p>
<p>This method of detecting <code>NaN</code> can be confusing and should be avoided. ES6 introduced a special function <code>Number.isNaN()</code>
which only returns <code>true</code> if the argument is <code>NaN</code> value. For clarity and consistency this function should be used to detect
<code>NaN</code> instead of all other methods.</p>

```ts
if (value !== value){ // Noncompliant: use Number.isNaN()
    processNaN(value);
}
```
<p>To fix your code replace self-comparison with <code>Number.isNaN()</code> function.</p>

```ts
if (Number.isNaN(value)){
    processNaN(value);
}
```
<p>Do not confuse <code>Number.isNaN()</code> with the legacy global <code>isNaN()</code> function. While they serve a similar purpose, the behavior
is very different - the global <code>isNaN()</code> tries to convert its argument into a number before checking if it is <code>NaN</code>. If the
argument cannot be converted into a number, <code>isNaN()</code> will return true, which may not be the desired behavior in all cases.</p>

```ts
isNaN('some text');          // true: Number('some text') returns NaN
Number.isNaN('some text');   // false: 'some text' is not NaN
```
<p>You should use the <code>Number.isNaN()</code> method over <code>isNaN()</code> to perform a strict check for <code>NaN</code> without any type
conversion.</p>
<p>If the intention was not to detect whether a value was NaN, fix the issue by not comparing the variable against itself.</p>

```ts
if (value !== anotherValue){
    // ...
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/NaN"><code>NaN</code></a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/isNaN"><code>Number.isNaN()</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/isNaN"><code>isNaN()</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Glossary/Type_Conversion">Type conversion</a> </li>
</ul>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S1764'>javascript:S1764</a> - Identical expressions should not be used on both sides of a binary operator </li>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S2688'>javascript:S2688</a> - "NaN" should not be used in comparisons </li>
</ul>