#!/bin/bash

#
# Checks a single Teamscale instance for problems.
# 
# Note that the first part (options parsing) is the same for all
# scripts here. This is intended to have self-contained scripts.
#

BASE_URL=
USER=
ACCESS_KEY=

function print_help {
    cat <<EOF
Options:
  -l, --url URL         The base URL of the Teamscale instance to check
  -u, --user USER       The user name to use for connecting
  -a, --accesskey KEY   The access key used for connecting
EOF
}

if [[ $# -lt 1 ]]
then
    print_help
    exit 3
fi

# Option parsing taken from https://stackoverflow.com/a/14203146/1237576
while [[ $# -gt 0 ]]
do
    key="$1"
    case $key in
	-l|--url)
	    BASE_URL="$2"
	    shift
	    ;;
	-u|--user)
	    USER="$2"
	    shift
	    ;;
	-a|--accesskey)
	    ACCESS_KEY="$2"
	    shift
	    ;;
	-h|--help)
	    print_help
	    exit 3
	    ;;
	*)
	    echo Unknown or unsupported option: $key
	    exit 3
	    ;;
    esac
    shift 
done

if [[ "$BASE_URL" = "" ]]
then
    echo Base URL not provided
    exit 3
fi

if [[ "$USER" = "" ]]
then
    echo User name not provided
    exit 3
fi

if [[ "$ACCESS_KEY" = "" ]]
then
    echo Access key not provided
    exit 3
fi

URL=$BASE_URL/api/health-check

RESPONSE=`curl -s -u$USER:$ACCESS_KEY $URL`
if [[ $? -ne 0 ]]
then
    echo Teamscale unreachable or invalid user credentials
    # rerun curl without "-s" to get error message
    curl -u$USER:$ACCESS_KEY $URL
    exit 3
fi

RC=`echo "$RESPONSE" | grep RC: | head -1 | tr -dc '0-9'`
if [[ "$RC" = "" ]]
then
    echo Could not parse return code!
    RC=3
fi

echo "$RESPONSE"
exit $RC

