# Dir 4.1 Run-time failures shall be minimized

## Category
Required

## Applies to
C90, C99

## Rationale
The C language was designed to provide very limited built-in run-time checking. While this approach allows generation of compact and fast executable code, it places the burden of run-time checking on the programmer. In order to achieve the desired level of robustness, it is therefore important that programmers carefully consider adding dynamic checks wherever there is potential for run-time errors to occur.

It is sometimes possible to demonstrate that the values of operands preclude the possibility of a run-time error during evaluation of an expression. In such cases, a dynamic check is not required provided that the argument supporting its omission is documented. Any such documentation should include the assumptions on which the argument depends. This information can be used during subsequent modifications of the code to ensure that the argument remains valid.

The techniques that will be employed to minimize run-time failures should be planned and documented, for example in design standards, test plans, static analysis configuration files and code review checklists. The nature of these techniques may depend on the integrity requirements of the project. See Section 5.2 for further details.

Note: the presence of a run-time error indicates a violation of Rule 1.3.

The following notes give some guidance on areas where consideration needs to be given to the provision of dynamic checks.
* arithmetic errors This includes errors occurring in the evaluation of expressions, such as overflow, underflow, divide by zero or loss of significant bits through shifting. In considering integer overflow, note that unsigned integer calculations do not strictly overflow but wrap around producing defined, but possibly unexpected, values. Careful consideration should be given to the ranges of values and order of operation in arithmetic expressions, for example:

```c
float32_t f1 = 1E38f;
float32_t f2 = 10.0f;
float32_t f3 = 0.1f;
float32_t f4 = ( f1 * f2 ) * f3;    /* (f1 * f2) will overflow */
float32_t f5 = f1 * ( f2 * f3 );    /* no overflow because (f2 * f3)
                                     * is (approximately) 1 */

if ( ( f3 >= 0.0f ) && ( f3 <= 1.0f ) )
{
  /*
   * no overflow because f3 is known to be in range 0..1 so the
   * result of the multiplication will fit in type float32_t
   */
  f4 = f3 * 100.0f;
}
```
* pointer arithmetic Ensure that when an address is calculated dynamically the computed address is reasonable and points somewhere meaningful. In particular it should be ensured that if a pointer points within an array, then when the pointer has been incremented or otherwise altered it still points within the same array. See restrictions on pointer arithmetic — Rule 18.1, Rule 18.2 and Rule 18.3.
* array bound errors Ensure that array indices are within the bounds of the array size before using them to index the array — Rule 18.1.
* function parameters The validity of arguments should be checked prior to passing them to library functions — Dir 4.11.
* pointer dereferencing Unless a pointer is already known to be non-NULL, a run-time check should be made before dereferencing that pointer. Once a check has been made, it is relatively straightforward within a single function to reason about whether the pointer may have changed and whether another check is therefore required. It is much more difficult to reason across function boundaries, especially when calling functions defined in other source files or libraries.

```c
/*
 * Given a pointer to a message, check the message header and return
 * a pointer to the body of the message or NULL if the message is
 * invalid.
 */
const char *msg_body ( const char *msg )
{
  const char *body = NULL;

  if ( msg != NULL )
  {
    if ( msg_header_valid ( msg ) )
    {
      body = &msg[ MSG_HEADER_SIZE ];
    }
  }
  return body;
}

  char msg_buffer[ MAX_MSG_SIZE ];
const char *payload;

payload = msg_body ( msg_buffer );

/* Check if there is a payload */
if ( payload != NULL )
{
  /* Process the payload */
}
```
* dynamic memory If dynamic memory allocation is being performed, it is essential to check that each allocation succeeds and that an appropriate degradation or recovery strategy is designed and tested.

## See also
Dir 4.11, Dir 4.12, Rule 1.3, Rule 18.1, Rule 18.2, Rule 18.3

---

Copyright The MISRA Consortium Limited © [Date - March 2013].
