# Rule 21.14 The Standard Library function *memcmp* shall not be used to compare null terminated strings

## Category
Required

## Analysis
Undecidable, System

## Applies to
C90, C99, C11

## Amplification
For the purposes of this rule, “null terminated strings” are:
*   String literals;
*   Arrays having *essentially character type* which contain a null character.

## Rationale
The Standard Library function `int memcmp ( const void *s1, const void *s2, size_t n );` performs a byte by byte comparison of the first `n` bytes of the two objects pointed at by `s1` and `s2`.

If *memcmp* is used to compare two strings and the length of either is less than `n`, then they may compare as different even when they are logically the same (i.e. each has the same sequence of characters before the null terminator) as the characters after a null terminator will be included in the comparison even though they do not form part of the string.

## Example
```c
extern char buffer1[ 12 ];
extern char buffer2[ 12 ];

void f1 ( void )
{
  ( void ) strcpy ( buffer1, "abc" );
  ( void ) strcpy ( buffer2, "abc" );

  /* The following use of memcmp is non-compliant */
  if ( memcmp ( buffer1, buffer2, sizeof ( buffer1 ) ) != 0 ) 
  {
    /*
     * The strings stored in buffer1 and buffer 2 are reported to be
     * different, but this may actually be due to differences in the
     * uninitialized characters stored after the null terminators.
     */
  }
}

/* The following definition violates other guidelines */
unsigned char headerStart[ 6 ] = { 'h', 'e', 'a', 'd', 0, 164 };

void f2 ( const uint8_t *packet )
{
  /* The following use of memcmp is compliant */
  if ( ( NULL != packet ) && ( memcmp( packet, headerStart, 6 ) == 0 ) )
  {
    /*
     * Comparison of values having essentially unsigned type reports that
     * contents are the same. Any null terminator is simply treated as a
     * zero value and any differences beyond it are significant.
     */
  }
}
```

## See also
Rule 21.15, Rule 21.16

---

Copyright The MISRA Consortium Limited © [Date - March 2025].