# Rule 9.4 An element of an object shall not be initialized more than once

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C99, C11

## Amplification
This rule applies to initializers for both objects and subobjects.

An aggregate initializer shall not contain two designators that refer to the same sub-object. An aggregate initializer shall not allow the *current object* to implicitly initialize an element that has been initialized previously in the initializer list.

## Rationale
The provision of *designated initializers* allows the naming of the components of an aggregate (structure or array) or of a union to be initialized within an initializer list and allows the object’s elements to be initialized in any order by specifying the array indices or structure member names they apply to (elements having no initialization value assume the default for uninitialized objects).

A designator can specify elements to be initialized in a different syntactic sequence from their order within the object layout. An initializer without a designator will always initialize the *next subobject* within the object layout.

Care is required when using *designated initializers* since the initialization of object elements can be inadvertently repeated. The C Standard specifies that the value produced by the syntactically-last initializer referring to an element in the list is used, overriding any preceding initializers for that element. The Standard leaves unspecified whether overridden initializers are evaluated, and therefore whether or not any *side effects* in the initializing expressions occur or not. This is not listed in Annex J of the C Standard.

In order to allow sparse arrays and structures, it is acceptable to only initialize those which are necessary to the application.

## Example
Array initialization:

```c
/*
 * Required behaviour using positional initialization
 * Compliant - a1 is -5, -4, -3, -2, -1
 */
int16_t a1[ 5 ] = { -5, -4, -3, -2, -1 };

/*
 * Similar behaviour using designated initializers
 * Compliant - a2 is -5, -4, -3, -2, -1
 */
int16_t a2[ 5 ] = { [ 0 ] = -5, [ 1 ] = -4, [ 2 ] = -3,
                    [ 3 ] = -2, [ 4 ] = -1 };

/*
 * Repeated designated initializer element values overwrite earlier ones
 * Non-compliant - a3 is -5, -4, -2, 0, -1
 */
int16_t a3[ 5 ] = { [ 0 ] = -5, [ 1 ] = -4, [ 2 ] = -3,
                    [ 2 ] = -2, [ 4 ] = -1 };
```

In the following non-compliant example, it is unspecified whether the *side effect* occurs or not:

```c
uint16_t *p;

void f ( void )
{
  uint16_t a[ 2 ] = { [ 0 ] = *p++, [ 0 ] = 1 };
}
```

Structure initialization:

```c
struct mystruct
{
  int32_t a;
  int32_t b;
  int32_t c;
  int32_t d;
};

/*
 * Required behaviour using positional initialization
 * Compliant - s1 is 100, -1, 42, 999
 */
struct mystruct s1 = { 100, -1, 42, 999 };

/*
 * Similar behaviour using designated initializers
 * Compliant - s2 is 100, -1, 42, 999
 */
struct mystruct s2 = { .a = 100, .b = -1, .c = 42, .d = 999 };

/*
 * Repeated designated initializer element values overwrite earlier ones
 * Non-compliant - s3 is 42, -1, 0, 999
 */
struct mystruct s3 = { .a = 100, .b = -1, .a = 42, .d = 999 };

/*
 * Positional initializer element values can overwrite earlier ones
 * if preceded by a designated element out of sequence
 * Non-compliant - s4 is 1, 4, 3, 0
 */
struct mystruct s4 = { .b = 2, .c = 3, .a = 1, /* b */ 4 };
```

## See also
Rule 9.6

---

Copyright The MISRA Consortium Limited © [Date - March 2025].