# Dir 4.9 A function should be used in preference to a function-like macro where they are interchangeable

## Category
Advisory

## Applies to
C90, C99, C11

## Amplification
This guideline applies only where a function is permitted by the syntax and *constraints* of the language standard.

## Rationale
In most circumstances, functions should be used instead of macros. Functions perform argument type-checking and evaluate their arguments once, thus avoiding problems with potential multiple *side effects*. In many debugging systems, it is easier to step through execution of a function than a macro. Nonetheless, macros may be useful in some circumstances.

Some of the factors that should be considered when deciding whether to use a function or a macro are:
*   The benefits of function argument and result type-checking;
*   The availability of *inline functions*, although note that the extent to which *inline* is acted upon is implementation-defined;
*   The trade-off between code size and execution speed;
*   Whether the possibility for compile-time evaluation is important: macros with constant arguments are more likely to be evaluated at compile-time than corresponding function calls;
*   Whether the arguments would be valid for a function: macro arguments are textual whereas function arguments are expressions;
*   Ease of understanding and maintainability.

## Exception
The use of *function-like macros* for `_Generic` selections is advised by Rule 23.1, and is therefore permitted by this Directive.

## Example
The following example is compliant. The *function-like macro* cannot be replaced with a function because it has a C operator as an argument:

```c
#define EVAL_BINOP( OP, L, R ) ( ( L ) OP ( R ) )

uint32_t x = EVAL_BINOP ( +, 1, 2 );
```

In the following example, the use of a macro to initialize an object with static storage duration is compliant because a function call is not permitted here:

```c
#define DIV2(X) ( ( X ) / 2 )

void f ( void )
{
  static uint16_t x = DIV2 ( 10 );  /* Compliant     - call not permitted */
  uint16_t y        = DIV2 ( 10 );  /* Non-compliant - call permitted     */
}
```

The following is compliant by exception — in most cases `_Generic` selections cannot be replaced with a function.

```c
#define GFUNC(X) ( _Generic( ... ) ) /* Compliant by exception */
```

## See also
Rule 13.2, Rule 20.7, Rule 23.1

---

Copyright The MISRA Consortium Limited © [Date - March 2025].