<h2>Why is this an issue?</h2>
<p>The <code>IDisposable</code> interface is a mechanism to release unmanaged resources, if not implemented correctly this could result in resource
leaks or more severe bugs.</p>
<p>This rule raises an issue when the recommended dispose pattern, as defined by Microsoft, is not adhered to. See the <strong>Compliant
Solution</strong> section for examples.</p>
<p>Satisfying the rule’s conditions will enable potential derived classes to correctly dispose the members of your class:</p>
<ul>
  <li> <code>sealed</code> classes are not checked. </li>
  <li> If a base class implements <code>IDisposable</code> your class should not have <code>IDisposable</code> in the list of its interfaces. In such
  cases it is recommended to override the base class’s <code>protected virtual void Dispose(bool)</code> method or its equivalent. </li>
  <li> The class should not implement <code>IDisposable</code> explicitly, e.g. the <code>Dispose()</code> method should be public. </li>
  <li> The class should contain <code>protected virtual void Dispose(bool)</code> method. This method allows the derived classes to correctly dispose
  the resources of this class. </li>
  <li> The content of the <code>Dispose()</code> method should be invocation of <code>Dispose(true)</code> followed by
  <code>GC.SuppressFinalize(this)</code> </li>
  <li> If the class has a finalizer, i.e. a destructor, the only code in its body should be a single invocation of <code>Dispose(false)</code>. </li>
  <li> If the class inherits from a class that implements <code>IDisposable</code> it must call the <code>Dispose</code>, or
  <code>Dispose(bool)</code> method of the base class from within its own implementation of <code>Dispose</code> or <code>Dispose(bool)</code>,
  respectively. This ensures that all resources from the base class are properly released. </li>
</ul>
<h3>Noncompliant code example</h3>
<pre>
public class Foo1 : IDisposable // Noncompliant - provide protected overridable implementation of Dispose(bool) on Foo or mark the type as sealed.
{
    public void Dispose() // Noncompliant - should contain only a call to Dispose(true) and then GC.SuppressFinalize(this)
    {
        // Cleanup
    }
}

public class Foo2 : IDisposable
{
    void IDisposable.Dispose() // Noncompliant - Dispose() should be public
    {
        Dispose(true);
        GC.SuppressFinalize(this);
    }

    public virtual void Dispose() // Noncompliant - Dispose() should be sealed
    {
        Dispose(true);
        GC.SuppressFinalize(this);
    }
}

public class Foo3 : IDisposable
{
    public void Dispose()
    {
        Dispose(true);
        GC.SuppressFinalize(this);
    }

    protected virtual void Dispose(bool disposing)
    {
        // Cleanup
    }

    ~Foo3() // Noncompliant - Modify Foo.~Foo() so that it calls Dispose(false) and then returns.
    {
        // Cleanup
    }
}{code}
</pre>
<h3>Compliant solution</h3>
<pre>
// Sealed class
public sealed class Foo1 : IDisposable
{
    public void Dispose()
    {
        // Cleanup
    }
}

// Simple implementation
public class Foo2 : IDisposable
{
    public void Dispose()
    {
        Dispose(true);
        GC.SuppressFinalize(this);
    }

    protected virtual void Dispose(bool disposing)
    {
        // Cleanup
    }
}

// Implementation with a finalizer
public class Foo3 : IDisposable
{
    public void Dispose()
    {
        Dispose(true);
        GC.SuppressFinalize(this);
    }

    protected virtual void Dispose(bool disposing)
    {
        // Cleanup
    }

    ~Foo3()
    {
        Dispose(false);
    }
}

// Base disposable class
public class Foo4 : DisposableBase
{
    protected override void Dispose(bool disposing)
    {
        // Cleanup
        // Do not forget to call base
        base.Dispose(disposing);
    }
}
</pre>
<h2>Resources</h2>
<p>Refer to</p>
<ul>
  <li> <a href="https://msdn.microsoft.com/en-us/library/498928w2.aspx">MSDN</a> for complete documentation on the dispose pattern. </li>
  <li> <a href="https://blog.stephencleary.com/2009/08/how-to-implement-idisposable-and.html">Stephen Cleary</a> for excellent Q&amp;A about
  IDisposable </li>
  <li> <a href="https://pragmateek.com/c-scope-your-global-state-changes-with-idisposable-and-the-using-statement/">Pragma Geek</a> for additional
  usages of IDisposable, beyond releasing resources. </li>
  <li> <a href="https://docs.microsoft.com/en-us/dotnet/api/system.idisposable?view=netframework-4.7">IDisposable documentation</a> </li>
</ul>

