<h2>Why is this an issue?</h2>
<p>Public fields in public classes do not respect the encapsulation principle and have three main disadvantages:</p>
<ul>
  <li> Additional behavior such as validation cannot be added. </li>
  <li> The internal representation is exposed, and cannot be changed afterwards. </li>
  <li> Member values are subject to change from anywhere in the code and may not meet the programmer’s assumptions. </li>
</ul>
<p>To prevent unauthorized modifications, private attributes and accessor methods (set and get) should be used.</p>
<h3>What is the potential impact?</h3>
<p>Public fields can be modified by any part of the code and this can lead to unexpected changes and hard-to-trace bugs.</p>
<p>Public fields don’t hide the implementation details. As a consequence, it is no longer possible to change how the data is stored internally without
impacting the client code of the class.</p>
<p>The code is harder to maintain.</p>
<h3>Exceptions</h3>
<p>This rule ignores <code>public final</code> fields because they are not modifiable. Also, annotated fields, whatever the annotation(s) will be
ignored, as annotations are often used by injection frameworks, which in exchange require having public fields.</p>
<h2>How to fix it</h2>
<p>Depending on your need there are multiple options:</p>
<ul>
  <li> Encapsulate the field
    <ol>
      <li> Make the field private. </li>
      <li> Define methods to get and set the value of the field.<br> These methods are commonly known as getter and setter methods and are prefixed by
      <code>get</code> and <code>set</code> followed by the name of the field. <em>Note:</em> as a bonus it is now possible to monitor value changes
      using breakpoints. </li>
    </ol>  </li>
  <li> Mark the field as <code>public final</code> if it is not supposed to change. </li>
</ul>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public class MyClass {

  public static final int SOME_CONSTANT = 0;     // Compliant - constants are not checked

  public String firstName;                       // Noncompliant

}
```
<h4>Compliant solution</h4>

```java
public class MyClass {

  public static final int SOME_CONSTANT = 0;     // Compliant - constants are not checked

  private String firstName;

  public String getFirstName() {
    return firstName;
  }

  public void setFirstName(String firstName) {
    this.firstName = firstName;
  }

}
```
<h3>How does this work?</h3>
<p>By having a setter and a getter the code can control how the field is accessed and modified. For example, adding validation in the setter method
will ensure that only valid values are set.</p>
<p>The access modifiers on the setter can also be changed to <code>private</code> or <code>protected</code> to restrain which code can modify the
value.</p>
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/493">CWE-493 - Critical Public Variable Without Final Modifier</a> </li>
</ul>