<h2>Why is this an issue?</h2>
<p>Many resources in Java need be closed after they have been used. If they are not, the garbage collector cannot reclaim the resources' memory, and
they are still considered to be in use by the operating system. Such resources are considered to be leaked, which can lead to performance issues.</p>
<p>Java 7 introduced the try-with-resources statement, which guarantees that the resource in question will be closed.</p>

```java
try (InputStream input = Files.newInputStream(path)) {
  // "input" will be closed after the execution of this block
}
```
<p>This syntax is safer than the traditional method using <code>try</code>, <code>catch</code>, and <code>finally</code> and hence should be
preferred.</p>
<p>This rule raises an issue if a closeable resources is not opened using a try-with-resources statement.</p>
<p>This rule is automatically disabled when the project’s <code>sonar.java.source</code> is lower than <code>7</code> as the close-with-resources
statement was unavailable prior to Java 7.</p>
<h2>How to fix it</h2>
<p>Use the try-with-resources syntax by moving the <code>Closable</code> variable declarations after the <code>try</code> keyword surrounded by
parentheses and separated by <code>;</code>:</p>

```java
try (/* resources declarations */) {
  // resources usage ...
}
```
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
FileReader fr = null;
BufferedReader br = null;

try { // Noncompliant, the FileReader and BufferedReader are instantiated without try-with-resources
  fr = new FileReader(fileName);
  br = new BufferedReader(fr);
  return br.readLine();
} catch (...) {
  ...
} finally {

  if (br != null) { // br has to be closed manually
    try {
      br.close();
    } catch(IOException e){...}
  }

  if (fr != null ) { // fr has to be closed manually
    try {
      br.close();
    } catch(IOException e){...}
  }

}
```
<h4>Compliant solution</h4>

```java
try ( // Compliant, all resources are instantiated within a try-with-resources statement and hence automatically closed after use
    FileReader fr = new FileReader(fileName);
    BufferedReader br = new BufferedReader(fr)
  ) {
  return br.readLine();
}
catch (...) {}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/6DZGBQ">CERT, ERR54-J.</a> - Use a try-with-resources statement to safely handle closeable
  resources </li>
  <li> <a href="https://docs.oracle.com/javase/tutorial/essential/exceptions/tryResourceClose.html">The Java™ Tutorials - The try-with-resources
  Statement</a> </li>
</ul>