<p>This rule raises an issue when a strict equality operator is used to compare objects of different types.</p>
<h2>Why is this an issue?</h2>
<p>The strict equality operator in JavaScript is represented by three equal signs (<code>===</code>), the strict inequality with (<code>!==</code>).
It is used to compare two values for equality, but with an important difference from the regular equality operator (<code>==</code>). The strict
equality operator compares both value and type, while the regular equality operator only compares values after performing type coercion if
necessary.</p>
<p>The problem with using the strict equality operator (<code>===</code>) with operands of dissimilar types lies in the way JavaScript handles the
comparison. When you use <code>===</code> to compare two values of different types, it will always return false since their types are different,
regardless of whether the values could be considered equal under certain conditions.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
let a = 8;
let b = "8";

if (a === b) { // Noncompliant: Always false since 'a' is a number and 'b' a string
  // ...
}
```
<h4>Compliant solution</h4>
<p>To address this issue, you can use the loose equality operator (<code>==</code>), which performs type coercion.</p>

```js
let a = 8;
let b = "8";

if (a == b) {
  // ...
}
```
<p>Alternatively, use the strict equality operator (<code>===</code>) but ensure that the operands have the same type before performing the
comparison. You can explicitly convert the operands to a common type using functions like <code>Number()</code>, <code>String()</code>, or other
appropriate methods depending on the situation.</p>

```js
let a = 8;
let b = "8";

if (a === Number(b)) {
  // ...
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Strict_equality">Strict equality
  (<code>===</code>)</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Strict_inequality">Strict inequality
  (<code>!==</code>)</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Equality">Equality (<code>==</code>)</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Inequality">Inequality (<code>!=</code>)</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Glossary/Type_coercion">Type coercion</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/Number"><code>Number()</code>
  constructor</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/String"><code>String()</code>
  constructor</a> </li>
  <li> MDN - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Boolean/Boolean"><code>Boolean()</code>
  constructor</a> </li>
</ul>