<h2>Why is this an issue?</h2>
<p>In React, <code>findDOMNode</code> is used to get the browser DOM node given a component instance. However, using <code>findDOMNode</code> is
fragile because the connection between the JSX node and the code manipulating the corresponding DOM node is not explicit. For example, changing the
external structure of returned JSX will affect the return value of <code>findDOMNode</code>. There are also other <a
href="https://react.dev/reference/react-dom/findDOMNode#caveats">caveats</a> when using <code>findDOMNode</code>.</p>

```ts
import { Component } from 'react';
import { findDOMNode } from 'react-dom';

class AutoselectingInput extends Component {
  componentDidMount() {
    const input = findDOMNode(this); // Noncompliant: findDOMNode is deprecated
    input.select();
  }

  render() {
    return <input defaultValue="Hello" />
  }
}
```
<p>Instead, one should get the component’s own DOM node from a ref. Pass your ref as the <code>ref</code> attribute to the JSX tag for which you want
to get the DOM node. This tells React to put this <code>&lt;input&gt;</code>’s DOM node into <code>inputRef.current</code>.</p>
<p>Use <code>createRef</code> to manage a specific DOM node. In modern React without class components, the equivalent code would call
<code>useRef</code> instead.</p>

```ts
import { createRef, Component } from 'react';

class AutoselectingInput extends Component {
  inputRef = createRef(null);

  componentDidMount() {
    const input = this.inputRef.current; // Always points to the input element
    input.select();
  }

  render() {
    return (
      <input ref={this.inputRef} defaultValue="Hello" />
    );
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/reference/react-dom/findDOMNode"><code>findDOMNode</code></a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/createRef"><code>createRef</code></a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/useRef"><code>useRef</code></a> </li>
  <li> React Documentation - <a href="https://react.dev/learn/manipulating-the-dom-with-refs">Manipulating the DOM with Refs</a> </li>
</ul>