<h2>Why is this an issue?</h2>
<p>React <code>isMounted()</code> is primarily used to avoid calling <code>setState()</code> after a component has unmounted, because calling
<code>setState()</code> after a component has unmounted will emit a warning. Checking <code>isMounted()</code> before calling <code>setState()</code>
does eliminate the warning, but it also defeats the purpose of the warning, which is raising awareness that the app is still holding a reference to
the component after the component has been unmounted.</p>
<p>When using ES6 classes, using <code>isMounted()</code> is already prohibited.</p>

```ts
class MyComponent extends React.Component {
  componentDidMount() {
    mydatastore.subscribe(this);
  }
  dataHandler() {
    if (this.isMounted()) { // Noncompliant: isMounted() hides the error
      //...
    }
  }
  render() {
    //... calls dataHandler()
  }
};
```
<p>Find places where <code>setState()</code> might be called after a component has unmounted, and fix them. Such situations most commonly occur due to
callbacks, when a component is waiting for some data and gets unmounted before the data arrives. Ideally, any callbacks should be canceled in
<code>componentWillUnmount</code>, before the component unmounts.</p>

```ts
class MyComponent extends React.Component {
  componentDidMount() {
    mydatastore.subscribe(this);
  }
  dataHandler() {
    //...
  }
  render() {
    //...
  }
  componentWillUnmount() {
    mydatastore.unsubscribe(this);
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://legacy.reactjs.org/blog/2015/12/16/ismounted-antipattern.html"><code>isMounted</code> is an
  Antipattern</a> </li>
</ul>