After compilation, string literals are stored in the cstring pool in the binary.
If many string literals share common substrings, these substrings are stored serveral times, unnecessarily bloating the binary size. 
To keep the binary size small, common substrings can be stored in separate string constants.

# What Does This Check Look For?
This check searches for parts of string literals that occur in several string literals.
It reports a finding if the length of the common part is above a configurable threshold and the potential binary-size savings of extracting the common part are above a configurable threshold.

# Why is This a Problem?
In some applications, the binary size of the generated program is limited by external factors (e.g., the available chip size).
To help keep the binary size below the limit, this check highlights optimization potential.  

In particular, extracting common substrings that are used across multiple code locations can significantly reduce the binary size.

# How Can I Resolve This?
1. Extract the common substring to a new string constant.
2. Concatenate the unique part of the string and the new constant.
# Examples
## Non-Compliant
```c++
// utils.cpp
#include "utils.h"
#include <cstdio> 
void foo() {
    printf("Prefix: This is a log statement.\n");
    printf("Prefix: Here another one.\n");
    printf("Prefix: And yet another log statement.\n");
    printf("Prefix: And a 4th one.\n");
}
```
## Compliant
```c++
// utils.cpp
#include "utils.h"
#include <cstdio> 
void foo() {
    const char* prefix = "Prefix: %s";
    printf(prefix, "This is a log statement.\n");
    printf(prefix, "Here another one.\n");
    printf(prefix, "And yet another log statement.\n");
    printf(prefix, "And a 4th one.\n");
}
```

# Finding Properties
* `Clone Length` The number of characters in the common substring.
* `Number of Clones` How many times this common substring is cloned.
* `Potential Savings` Estimated number of bytes that can be saved by extracting the common substring. See next section for details.

# Computation of Potential Savings
To estimate how many bytes can be saved by extracting the common substring, we assume
1. that the common substring is stored once at a new location (i.e., binary size increases by the common substring plus one null terminator character)
2. that the common substring is not stored at the same locations anymore (i.e., binary size decreases by common-substring length multiplied by the number of occurrences)
3. that one additional null terminator is required at common-substring locations where the string literal does not begin with the common substring
4. that additional operations are required at each common-substring location (there are many possible ways to concatenate the strings, so we assume an average cost of 1 for this)