Using String formatting in log statements can degrade performance.
Consider using parameterized logging methods provided by the logging framework instead.

# What Does This Check Look For?

This check identifies instances where String formatting operations like `String.format` or string concatenation are used within log statement calls.

# Why is This a Problem?

Using `String.format` or string concatenation within log statements can introduce unnecessary performance overhead. This is particularly problematic for disabled log levels, where the formatted message will never be used, yet the computational cost of formatting the string is still incurred. Such operations can lead to decreased application performance, especially in scenarios requiring extensive logging.

# How Can I Resolve This?

Instead of formatting strings directly in the log statement, leverage the logging framework's built-in parameterized logging capability. Most Java logging frameworks support this feature, which defers the formatting operation until it is confirmed that the log statement is enabled. This approach minimizes the performance impact by avoiding needless string formatting operations.

# Examples

## Non-Compliant

```java
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Slf4jExample {

	private static Logger LOGGER = LoggerFactory.getLogger(Slf4jExample.class);

	public static void main(String[] args) {
		LOGGER.info(String.format("Hello %s", args[1]));
	}
}
```

## Compliant

```java
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Slf4jExample {

	private static Logger LOGGER = LoggerFactory.getLogger(Slf4jExample.class);

	public static void main(String[] args) {
		LOGGER.info("Hello {}", args[1]);
	}
}
```