Certain exceptions should not be caught, because it is considered bad practise.
Catching such exceptions should be avoided.

# What Does This Check Look For?
This check identifies catch clauses that catch certain exceptions.
The list of exceptions to avoid catching can be customized.
By default, this list includes classes such as `ClassCastException`, `NullPointerException`, or `ArithmeticException`.

# Why is This a Problem?
Catching exceptions like `ClassCastException`, `NullPointerException`, or `ArithmeticException` is unusual in normal code.
These exceptions usually signal programming errors or logical flaws that should be addressed directly in the code rather than being circumvented through exception handling.
Ignoring the root cause of such exceptions and relying on catch blocks for control flow can lead to code that is less reliable, more difficult to debug, and potentially more vulnerable to errors.

# How Can I Resolve This?
The Problem can be resolved by fixing the underlying issue or using checks instead of exceptions.

# Examples
## Non-Compliant
A `NullPointerException` will be thrown by `list.get(0)`, if `getFirst()` is called before `setList()`.
```java
class Foo {

	private List<String> list;

	String getFirst() {
		try {
			return list.get(0);
		} catch (NullPointerException e) {
			// handle exception
		}
	}

	void setList(List<String> list) {
		this.list = list;
	}
}
```

## Compliant
With inserting a null check before `list.get(0)`.
```java
class Foo {

	private List<String> list;

	String getFirst() {
		if(list == null) {
		    // handle null value
		}
		return list.get(0);
	}

	void setList(List<String> list) {
		this.list = list;
	}
}
```
Even better, directly fix the underlying problem.
In this example by initializing the list in the constructor.

```java
class Foo {

	private final List<String> list;
	
	Foo(List<String> list) {
		this.list = list;
	}

	String getFirst() {
		return list.get(0);
	}

	void addToList(List<String> list) {
		this.list.addAll(list);
	}
}
```

# Where Can I Learn More?
- [Refactoring Guru: Replace Exception with Test](https://refactoring.guru/replace-exception-with-test)