Omitting a `default` case in a `switch` statement may leave some cases unhandled. 
Consider adding a `default` case to handle unexpected or future values, or to explicitly document that no action is necessary.

# What Does This Check Look For?
This check detects any `switch` statement that misses a `default` case.

_Note: If the option `ignore missing default cases in switch statements over enums` is enabled than this check will not flag switch statements over enums._ 

# Why is This a Problem?
Without a `default` case, a `switch` statement might silently ignore values that do not match any of the explicitly handled cases.
This omission can make the code error-prone and harder to maintain, as all possible cases might not be visibly accounted for, especially if the range of values changes in the future.
Including a `default` case, even without containing a statement, improves the code completeness and readability by making explicit that all possible cases have been considered.

It is worth noting that this check fosters defensive programming, but might not be appropriate for all projects. 

# How Can I Resolve This?
Add a `default` case to the `switch` statement. 
The action taken in the `default` case can vary. 
Ideally, any unexpected values should be handled directly or at least logged if nothing else is to be done.
If applicable, it can also be appropriate to throw an exception or assert that this path can never execute: `assert(false); // can never happen`.
Even if no action needs to be taken, the `default` case serves as an explicit indication that no action is required.
Some coding guidelines suggest to add a comment in this case that explicitly documents why nothing needs to be done.

The practice of always adding a `default` case ensures that the code behaves predictably even when encountering unexpected values and facilitates future maintenance by signaling to developers that all possible cases have been considered.

# Examples
## Non-Compliant
```C
switch (grade) {
   case 'A':
      addToAList();
      break;
   case 'B':
      addToBList();
      break;
   case 'F':
      addToFailList();
      break;	   
      // No default case
      // Not clear if other grades should be ignored
      // or be handled  
}
```
```C
enum Status
{ DONE, ONGOING, ARCHIVED } state;

switch (state) {
   case DONE:
      nextState = ARCHIVED;
      break;
   case ONGOING:
      nextState = DONE;
      break;
   case ARCHIVED:
      nextState = ONGOING;
      break;	   
      // No default case
      // What if new values get added to the enum?   
}
```
## Compliant
```C
switch (grade) {
   case 'A':
      addToAList();
      break;
   case 'B':
      addToBList();
      break;
   case 'F':
      addToFailList();
      break;	   
   default: 
      log_error('Unexpected grade');
      // Or add a comment such as
      // All other letter grades besides A, B, and F should be
      // ignored. No need to throw exception.
      break;
}
```

```C
enum Status
{ DONE, ONGOING, ARCHIVED } state;

switch (state) {
   case DONE:
      nextState = ARCHIVED;
      break;
   case ONGOING:
      nextState = DONE;
      break;
   case ARCHIVED:
      nextState = ONGOING;
      break;	    	
   default:
      // Handle unexpected values here
      log("Disallowed status");
      // And if applicable, ensure this should never happen
      assert(false);
      break;   
}
```

# Where Can I Learn More?
- Guideline: MISRA C:2012: 16.4 - Every switch statement shall have a default label
- [CPP Core Guidelines: ES.79: Use default to handle common cases (only)](https://isocpp.github.io/CppCoreGuidelines/CppCoreGuidelines#es79-use-default-to-handle-common-cases-only)
