Writing error-free code when destructors can fail is very difficult.
Mark them as `noexcept` to be safe.

# What Does This Check Look For?

This check detects [destructors](https://en.cppreference.com/w/cpp/language/destructor) that are not declared with the [`noexcept` specifier](https://en.cppreference.com/w/cpp/language/noexcept_spec).

# Why is This a Problem?

It is very difficult to write error-free code when destructors are allowed to fail by throwing an [exception](https://en.cppreference.com/w/cpp/language/exceptions).

As an example, consider the following situation:
1. Somewhere `throw some_exception()` is called.
2. Stack unwinding begins and all stack frames between the `throw` and `catch (some_exception& e)` are popped.
3. This process causes the destruction of all local objects in those stack frames.
4. One of these object's destructors throws another exception `other_exception`.

Now there is an ambiguity:
Should the unwinding process ignore the new exception and continue towards the `catch (some_exception& e)`?
Or should it unwind towards a `catch (other_exception& e)` handler for the new exception?
Both variants loose information, so the C++ language specification mandates that `terminate()` is called instead, which kills the entire process.

# How Can I Resolve This?

While it is technically legal to throw an exception from a destructor, as long as it doesn't happen while processing another exception, this condition is very difficult to enforce.
The safe way out is to never throw an exception from a destructor, which is exactly what `noexcept` enforces.

Finally, marking destructors as `noexcept` prevents them from accidentally being poisoned by a base or member class whose destructor is `noexcept(false)`.

# Examples
## Non-Compliant

``` cpp
struct A {
    ~A() { }
};
```

## Compliant

``` cpp
struct A {
    ~A() noexcept(true) { }
    // or for short
    ~A() noexcept { }
};
```

# Where Can I Learn More?

* [Guideline: C++ Core Guidelines: C.36 - A destructor must not fail](https://github.com/isocpp/CppCoreGuidelines/blob/master/CppCoreGuidelines.md#c36-a-destructor-must-not-fail)
* [Guideline: C++ Core Guidelines: C.37 - Make destructors noexcept](https://github.com/isocpp/CppCoreGuidelines/blob/master/CppCoreGuidelines.md#Rc-dtor-noexcept).
* [ISO C++ FAQ: How can I handle a destructor that fails?](https://isocpp.org/wiki/faq/exceptions#dtors-shouldnt-throw)