<h2>Why is this an issue?</h2>
<p>A non-static inner class has a reference to its outer class, and access to the outer class' fields and methods. That class reference makes the
inner class larger and could cause the outer class instance to live in memory longer than necessary.</p>
<p>If the reference to the outer class isn’t used, it is more efficient to make the inner class <code>static</code> (also called nested). If the
reference is used only in the class constructor, then explicitly pass a class reference to the constructor. If the inner class is anonymous, it will
also be necessary to name it.</p>
<p>However, while a nested/<code>static</code> class would be more efficient, it’s worth noting that there are semantic differences between an inner
class and a nested one:</p>
<ul>
  <li> an inner class can only be instantiated within the context of an instance of the outer class. </li>
  <li> a nested (<code>static</code>) class can be instantiated independently of the outer class. </li>
</ul>
<h2>How to fix it</h2>
<p>There are two scenarios in which this rule will raise an issue:</p>
<ol>
  <li> On an <em>inner class</em>: make it <code>static</code>. </li>
  <li> On a <em>local class</em>: extract it as a <code>static</code> <em>inner class</em>. </li>
</ol>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>Inner classes that don’t use the outer class reference should be static.</p>

```java
public class Fruit {
  // ...

  public class Seed {  // Noncompliant; there's no use of the outer class reference so make it static
    int germinationDays = 0;
    public Seed(int germinationDays) {
      this.germinationDays = germinationDays;
    }
    public int getGerminationDays() {
      return germinationDays;
    }
  }
}
```
<h4>Compliant solution</h4>

```java
public class Fruit {
  // ...

  public static class Seed {
    int germinationDays = 0;
    public Seed(int germinationDays) {
      this.germinationDays = germinationDays;
    }
    public int getGerminationDays() {
      return germinationDays;
    }
  }
}
```
<p>Local classes that don’t use the outer class reference should be extracted as a static inner classes.</p>
<h4>Noncompliant code example</h4>

```java
public class Foo {
  public Foo() {
    class Bar { // Noncompliant
      void doSomething() {
        // ...
      }
    }
    new Bar().doSomething();
  }

  public void method() {
    class Baz { // Noncompliant
      void doSomething() {
        // ...
      }
    }
    new Baz().doSomething();
  }
}
```
<h4>Compliant solution</h4>

```java
public class Foo {
  public Foo() {
    new Bar().doSomething();
  }

  public void method()  {
    new Baz().doSomething();
  }

  private static class Bar { // Compliant
    void doSomething() {
      // ...
    }
  }

  private static class Baz { // Compliant
    void doSomething() {
      // ...
    }
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/javase/tutorial/java/javaOO/nested.html">Oracle Java SE - Nested Classes</a> </li>
  <li> <a href="https://docs.oracle.com/javase/tutorial/java/javaOO/localclasses.html">Oracle Java SE - Local Classes</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.geeksforgeeks.org/difference-between-static-and-non-static-nested-class-in-java/">GeeksforGeeks - Difference between
  static and non-static nested class in Java</a> </li>
</ul>