# Rule 0.2.3 Types with *limited visibility* should be *used* at least once

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Amplification
A type has *limited visibility* if it is declared in block scope or in unnamed namespace scope.

For the purposes of this rule:
*   Type aliases, primary class templates, and alias templates are considered types.
*   The closure type associated with a lambda is always *used*.
*   A type is *used* if it is referenced within the *translation unit* outside of its definition.
*   An enumeration type is *used* if any of its enumerators are *used*.
*   An anonymous union is *used* if any of its members are *used*.
*   The definition of a type includes the definition of its members and hidden friends.
*   The definition of a class template includes its partial and explicit specializations.

## Rationale
If a type is declared but not *used*, then it is unclear to a reviewer if the type is redundant or it has been
left *unused* by mistake.

## Exception
This rule does not apply to:
1.  Types that have at least one *declaration* with the `[[maybe_unused]]` attribute.
2.  Template parameters.
3.  Partial or explicit specializations of class templates.

## Example
```cpp
int16_t f1()
{
  using T1 = int16_t; // Non-compliant
  using T2 [[maybe_unused]] = int32_t; // Compliant by exception #1

  return 67;
}

namespace
{
  struct A1 { A1 f(); }; // Compliant
  struct A2 { A2 f(); }; // Non-compliant

  struct A2; // Not a use of A2

  A2 A2::f() { return *this; } // Not a use of A2

  template< typename T > // Compliant by exception #2
  void foo()
  {
    A1 a; // Use of A1
    a.f(); // - even if foo is not instantiated
  }
}

template< bool cond >
inline auto foo()
{
  struct res { int32_t i; }; // Compliant

  if constexpr ( cond )
  {
    return 42;
  }
  else
  {
    return res { 42 }; // res is utilized, even if cond is true
  }
}

template< typename >
int32_t bar()
{
  return 42;
}

int32_t f2()
{
  return bar< struct P >(); // Compliant - P is used
}

namespace
{
  template< typename > struct C1 {}; // Non-compliant
                                     // - C1 only utilized in its definition

  template<> struct C1< int32_t > // Compliant by exception #3
  {
    void mbr()
    {
      C1< char > cc;
    }
  };
}

namespace
{
  template< typename > struct C2 {}; // Compliant - C2< float > used

  template<> struct C2< int32_t >; // Compliant by exception #3

  C2< float > cf; // Use of C2
}

namespace
{
  static union // Non-compliant
  {
    int32_t i1;
    int32_t j1;
  };

  static union // Compliant
  {
    int32_t i2;
    int32_t j2;
  };
}

void f3()
{
  ++i2; // Uses the anonymous union holding i2
}

namespace
{
  void f4()
  {
    []( auto ){}; // Compliant - closure type is always used
  }
}
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
