<h2>Why is this an issue?</h2>
<p>There is no requirement that class names be unique, only that they be unique within a package. Therefore trying to determine an object’s type based
on its class name is an exercise fraught with danger. One of those dangers is that a malicious user will send objects of the same name as the trusted
class and thereby gain trusted access.</p>
<p>Instead, the <code>instanceof</code> operator or the <code>Class.isAssignableFrom()</code> method should be used to check the object’s underlying
type.</p>
<h3>Noncompliant code example</h3>

```java
package computer;
class Pear extends Laptop { ... }

package food;
class Pear extends Fruit { ... }

class Store {

  public boolean hasSellByDate(Object item) {
    if ("Pear".equals(item.getClass().getSimpleName())) {  // Noncompliant
      return true;  // Results in throwing away week-old computers
    }
    return false;
  }

  public boolean isList(Class<T> valueClass) {
    if (List.class.getName().equals(valueClass.getName())) {  // Noncompliant
      return true;
    }
    return false;
  }
}
```
<h3>Compliant solution</h3>

```java
class Store {

  public boolean hasSellByDate(Object item) {
    if (item instanceof food.Pear) {
      return true;
    }
    return false;
  }

  public boolean isList(Class<T> valueClass) {
    if (valueClass.isAssignableFrom(List.class)) {
      return true;
    }
    return false;
  }
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/486">CWE-486 - Comparison of Classes by Name</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/eDdGBQ">CERT, OBJ09-J.</a> - Compare classes and not class names </li>
</ul>