<h2>Why is this an issue?</h2>
<p>Assertions are statements that check whether certain conditions are true. They are used to validate that the actual results of a code snippet match
the expected outcomes. By using assertions, developers can ensure that their code behaves as intended and identify potential bugs or issues early in
the development process.</p>
<p>An incomplete assertion refers to a situation where an assertion is written but lacks some necessary components or conditions, making it
insufficient to fully validate the expected behavior of the code being tested. Writing incomplete assertions can lead to false positives or false
negatives in your test suite, making it less reliable.</p>
<p>This rule checks for incomplete assertions with Chai.js in the following cases:</p>
<ul>
  <li> When <code>assert.fail</code>, <code>expect.fail</code> or <code>should.fail</code> are present but not called. </li>
  <li> When an <code>expect(...)</code> or <code>should</code> assertion is not followed by an assertion method, such as <code>equal</code>. </li>
  <li> When an <code>expect</code> or <code>should</code> assertion ends with a <a
  href="https://www.chaijs.com/api/bdd/#method_language-chains">chainable getters</a>, such as <code>.that</code>, or a modifier, such as
  <code>.deep</code>. </li>
  <li> When an <code>expect</code> or <code>should</code> assertion function, such as <code>.throw</code>, is not called. </li>
</ul>
<p>In such cases, what is intended to be an assertion doesn’t actually assert anything.</p>

```js
const assert = require('chai').assert;
const expect = require('chai').expect;

describe("incomplete assertions", function() {
    const value = 42;

    it("uses chai 'assert'", function() {
        assert.fail; // Noncompliant: Missing the call to 'fail'
    });

    it("uses chai 'expect'", function() {
        expect(1 == 1); // Noncompliant: Should chain with 'to.equal'
        expect(value.toString).to.throw; // Noncompliant: Missing the type of the exception
    });
});
```
<p>Make sure to write complete and precise assertions. Always include the necessary comparison methods (e.g., <code>.to.equal()</code>,
<code>.to.be.true</code>, etc.) to make the expectations clear and leave no room for ambiguity.</p>

```js
const assert = require('chai').assert;
const expect = require('chai').expect;

describe("complete assertions", function() {
    const value = 42;

    it("uses chai 'assert'", function() {
        assert.fail();
    });

    it("uses chai 'expect'", function() {
        expect(1).to.equal(1);
        expect(value.toString).to.throw(TypeError);
    });
});
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Chai.js Documentation - <a href="https://www.chaijs.com/api/assert">Assert</a> </li>
  <li> Chai.js Documentation - <a href="https://www.chaijs.com/api/bdd"><code>expect</code> and <code>should</code></a> </li>
</ul>