<h2>Why is this an issue?</h2>
<p>In JavaScript, a generator is a special type of function that can be paused and resumed during its execution. It allows you to define an iterative
algorithm by writing a function that can maintain its internal state and produce a sequence of values over time.</p>
<p>Generators are defined using a function syntax with an asterisk <code>(*)</code> appended to the <code>function</code> keyword
(<code>function*</code>). Within the generator function, you can use the <code>yield</code> keyword to produce a value and temporarily pause the
execution of the function, returning that value to the consumer.</p>

```js
function* generate() {
  yield 1;
  yield 2;
  yield 3;
}
```
<p>This example defines a generator function named <code>generate</code> that produces a sequence of values: 1, 2, and 3.</p>
<p>Using a generator without the <code>yield</code> keyword can limit the usefulness and potential benefits of generators. When you use the
<code>yield</code> keyword without providing a value, it creates a yield expression that pauses the execution of the generator function and returns
<code>undefined</code> as the yielded value.</p>

```js
function* range(start, end) {
  while (start < end) {
    yield; // Noncompliant: The generator yields undefined
    start++;
  }
}
```
<p>Yielding without a value makes it harder for the generator consumer to understand the purpose or context of the yielded value. Instead, one should
always provide an explicit value with <code>yield</code> (using <code>undefined</code> when that is the intention) to make the generated sequence more
meaningful and informative.</p>

```js
function* range(start, end) {
  while (start < end) {
    yield start;
    start++;
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Generator">Generator</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/yield"><code>yield</code></a> </li>
</ul>