Using comparison operators to check for `NaN` does not work.
Use dedicated language-specific functions to check for `NaN` instead.

# What Does This Check Look For?
This check detects comparisons with `NaN`.

# Why is This a Problem?
Usually, `NaN` is returned by functions to indicate invalid operations or values that cannot be represented (e.g. complex numbers like the square root or the logarithm of -1).
Hence, the callers of those functions should check whether the returned value is a valid number or `NaN`.

Checking for `NaN` using common comparators like `<`, `==` etc. is **does not work**, in particular checking equality of `NaN` with itself results in `false`.
Yet, the program will treat `NaN` as a normal number while calculating with the double/float that is `NaN`. 
As a result, this calculation will **not** raise an exception.

This way, `NaN` return values might not be identified, which may lead to **bugs** or **unwanted behavior** (e.g. adding a number to `NaN` results in `NaN`).

# How Can I Resolve This?
To check for `NaN`, you can use language specific functions:
- For Java, use `Double.isNaN(double foo)` or `Float.isNaN(float foo)`
- For Python, use `math.isnan(x)`
- For C#, use `Double.IsNaN(Double)` or `Single.IsNaN(Single)`
- For C++, use `std::isnan(x)` from `<cmath>`
- For C, use `isnan(x)` from `math.h`
- For JavaScript, use `Number.isNaN(x)`

# Examples

## Java

### Non-Compliant

```java
if(foo == Double.NaN) {    // non-compliant: check will always return false
    someCode();
}
```

### Compliant
```java
if(Double.isNaN(foo)) {
    someCode();
}
```

## Python

### Non-Compliant

```python
def f(a):
    return a == float('nan')    // non-compliant: check will always return false
```

### Compliant

```python
import math

def f(a):
    return math.isnan(a)
```

# Where Can I Learn More?
- [Javatpoint: Java Double isNaN() Method](https://www.javatpoint.com/java-double-isnan-method)
- [Baeldung: NaN in Java](https://www.baeldung.com/java-not-a-number#comparison)
- [C++ reference: std::isnan](https://en.cppreference.com/w/cpp/numeric/math/isnan)
