Declaring parameters as an `IN OUT` when they are only meant to function as `IN` parameters can create unexpected behavior and potentially introduce bugs in the code. 
This happens because when parameters are declared as `IN OUT`, they are expected to take in a value and return an output based on that input. 
When such parameters are only used as `IN` parameters, it can lead to confusion for developers who might not understand the parameter's true functionality, resulting in incorrect code being written.

This process identifies routines where `IN OUT` parameters are declared, but are used only as `IN` parameters.

# Why is This a Problem?
Declaring an `IN` parameter as `IN OUT` can lead to several problems, including:

* Incorrect usage: Developers may mistakenly assume that an `IN` parameter declared as `IN OUT` can be used as an output, leading to incorrect assignments and unexpected behavior.
* Difficulty tracking changes: Since changes made to an `IN` parameter declared as `IN OUT` are reflected in the caller's scope, it can be difficult to track the validity and consistency of the data being passed around.
* Potential for confusion: The inconsistent usage of the parameter can lead to confusion and bugs, especially when code is reviewed or modified by multiple developers.

# How Can I Resolve This?
To prevent potential confusion and errors arising from misdeclaring IN parameters as `IN OUT` parameters, the following guidelines can be helpful:

* Understand the usage: Before declaring a parameter, it is vital to know its exact role. If it is not meant to modify the input, declare it as `IN` only.
* Select the correct mode: Ensure to choose the correct mode that accurately describes the parameter's intended use.
* Thoroughly review code: Carefully examine the parameter usage when reviewing code to ensure consistency and coherence, which will help minimize potential bugs.

## Example (before)
```oraclesqlplus
DECLARE
PROCEDURE sample_procedure(p_parameter IN OUT VARCHAR2) IS
BEGIN
    -- using only the input value and not modifying it
    DBMS_OUTPUT.PUT_LINE('Input Value: ' || p_parameter);
END sample_procedure;
```

## Example (after)
```oraclesqlplus
DECLARE
PROCEDURE sample_procedure(p_parameter IN VARCHAR2) IS
BEGIN
    DBMS_OUTPUT.PUT_LINE('Input Value: ' || p_parameter);
END sample_procedure;
```

# Where Can I Learn More?

- [G-7170: Avoid using an IN OUT parameter as IN or OUT only](https://trivadis.github.io/plsql-and-sql-coding-guidelines/v4.2/4-language-usage/7-stored-objects/1-general/g-7170/)



