<h2>Why is this an issue?</h2>
<p>Whenever the <code>value</code> property of React context changes, React will rerender the context and all its child nodes and consumers. In
JavaScript, things like object literals or function expressions will create a new identity every time they are evaluated. Such constructions should
not be directly used as context <code>value</code> because React will always consider they have changed. This can significantly impact
performance.</p>

```ts
function Component() {
  return (
    <SomeContext.Provider value={{foo: 'bar'}}> { /* Noncompliant: value is an object literal */ }
      <SomeComponent />
    </SomeContext.Provider>
  );
}
```
<p>To avoid additional rerenders wrap the value in a <code>useMemo</code> hook. Use the <code>useCallback()</code> hook if the value is a
function.</p>

```ts
function Component() {
  const obj = useMemo(() => ({foo: 'bar'}), []); // value is cached by useMemo
  return (
    <SomeContext.Provider value={obj}> { /* Compliant */ }
      <SomeComponent />
    </SomeContext.Provider>
  );
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://reactjs.org/docs/context.html#caveats">Context Caveats</a> </li>
  <li> React Documentation - <a href="https://react.dev/learn/passing-data-deeply-with-context">Passing Data Deeply with Context</a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/useCallback"><code>useCallback</code> hook</a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/useMemo"><code>useMemo</code> hook</a> </li>
</ul>