/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.issues.radar.client;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.teamscale.commons.service.client.ServiceCallException;
import com.teamscale.core.authenticate.index.AccessTokenIndex;
import com.teamscale.core.option.server.ServerOptionIndex;
import com.teamscale.core.rest.client.Retrofit;
import com.teamscale.core.rest.client.authentication.ICustomAuthenticationInterceptor;
import com.teamscale.index.apple.idms.A3Authenticator;
import com.teamscale.index.issues.BugTrackerException;
import com.teamscale.index.issues.radar.client.IRadarApi;
import com.teamscale.index.issues.radar.client.RadarAuthenticationInterceptor;
import com.teamscale.index.issues.radar.client.RadarComponentNameQuery;
import com.teamscale.index.issues.radar.client.RadarComponentQuery;
import com.teamscale.index.issues.radar.client.RadarFindProblemIDsQuery;
import com.teamscale.index.issues.radar.client.RadarKeywordQuery;
import com.teamscale.index.issues.radar.model.RadarAuthenticationToken;
import com.teamscale.index.issues.radar.model.RadarComponent;
import com.teamscale.index.issues.radar.model.RadarComponentBundle;
import com.teamscale.index.issues.radar.model.RadarComponentBundleGroup;
import com.teamscale.index.issues.radar.model.RadarKeyword;
import com.teamscale.index.issues.radar.model.RadarProblem;
import java.io.IOException;
import java.time.format.DateTimeFormatter;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import okhttp3.ResponseBody;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.conqat.engine.commons.util.JsonSerializationException;
import org.conqat.engine.commons.util.JsonUtils;
import org.conqat.engine.persistence.store.StorageException;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.collections.CounterSet;
import org.jspecify.annotations.NonNull;

public class RadarClient {
    public static final DateTimeFormatter RADAR_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssxx");
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String STATUS_RESPONSE_FIELD = "status";
    private static final int PROBLEM_BATCH_SIZE = 15;
    private static final int MAX_ATTEMPTS = 3;
    private final IRadarApi radar;
    private final Pattern errorStatusPattern = Pattern.compile("^[45]\\d\\d\\W.*");
    private final RadarAuthenticator authenticator;

    private RadarClient(IRadarApi radar, RadarAuthenticator authenticator) {
        this.radar = radar;
        this.authenticator = authenticator;
    }

    public static RadarClient create(AccessTokenIndex accessTokenIndex, ServerOptionIndex serverOptionIndex, String appleA3MultiOptionIdentifier, Logger interactionLogger) throws StorageException {
        A3Authenticator a3authenticator = A3Authenticator.create(accessTokenIndex, serverOptionIndex, appleA3MultiOptionIdentifier);
        RadarAuthenticator radarAuthenticator = new RadarAuthenticator(a3authenticator);
        String radarBaseUrl = a3authenticator.getRadarBaseUrl();
        IRadarApi api = (IRadarApi)Retrofit.builder((String)radarBaseUrl).withCustomAuthenticationInterceptor((ICustomAuthenticationInterceptor)new RadarAuthenticationInterceptor(radarBaseUrl, accessTokenIndex, radarAuthenticator)).withInteractionLogger(interactionLogger).create(IRadarApi.class);
        return new RadarClient(api, radarAuthenticator);
    }

    private static Map<String, String> generateHeadersForGetProblem(List<String> customFields) {
        List<String> defaultAttributes = Arrays.stream(RadarProblem.class.getDeclaredFields()).filter(field -> field.isAnnotationPresent(JsonProperty.class)).map(field -> {
            String annotationValue = field.getAnnotation(JsonProperty.class).value();
            if (annotationValue.isEmpty()) {
                return field.getName();
            }
            return annotationValue;
        }).toList();
        Preconditions.checkState((boolean)Collections.disjoint(defaultAttributes, customFields), (Object)"Custom fields must never request default attributes");
        return Map.of("X-Fields-Requested", String.join((CharSequence)",", CollectionUtils.concatenateLists(defaultAttributes, customFields, (List[])new List[0])));
    }

    @CanIgnoreReturnValue
    public RadarAuthenticationToken requestAuthenticationToken() throws ServiceCallException {
        return this.authenticator.requestAuthenticationToken();
    }

    public List<String> findProblemIds(List<String> components, List<String> componentBundles, List<String> componentBundleGroups, long minCreationTimestamp, long minModificationTimestamp, Set<String> keywords, boolean inclusiveDateMatching) throws BugTrackerException {
        List<Long> bundleGroupIds = this.findBundleGroupIds(componentBundleGroups);
        RadarFindProblemIDsQuery query = RadarFindProblemIDsQuery.idsOfProblemsChangedAfter(components, componentBundles, bundleGroupIds, minCreationTimestamp, minModificationTimestamp, keywords, inclusiveDateMatching);
        try {
            return CollectionUtils.map(this.radar.findProblems(query), String::valueOf);
        }
        catch (ServiceCallException e) {
            throw new BugTrackerException("Could not retrieve Problem IDs to synchronize", e);
        }
    }

    public List<RadarComponentBundle> findComponentBundles(String componentBundleName) throws BugTrackerException {
        try {
            return this.radar.findComponentBundles(new RadarComponentNameQuery(componentBundleName));
        }
        catch (ServiceCallException e) {
            throw new BugTrackerException("Error finding bundle: " + componentBundleName, e);
        }
    }

    public List<RadarComponentBundleGroup> findComponentBundleGroups(String componentBundleGroupName) throws BugTrackerException {
        try {
            return this.radar.findComponentBundleGroups(new RadarComponentNameQuery(componentBundleGroupName));
        }
        catch (ServiceCallException e) {
            throw new BugTrackerException("Cannot find bundle group: " + componentBundleGroupName, e);
        }
    }

    public List<RadarKeyword> findKeyword(String keyword) throws BugTrackerException {
        try {
            return this.radar.findKeyword(new RadarKeywordQuery(keyword));
        }
        catch (ServiceCallException e) {
            throw new BugTrackerException("Error retrieving keyword", e);
        }
    }

    public RadarComponent getComponent(String componentString) throws BugTrackerException {
        try {
            RadarComponentQuery componentQuery = RadarComponent.createQueryFromComponentName(componentString);
            String version = componentQuery.version();
            if (version == null) {
                version = "all";
            }
            return this.radar.getComponent(componentQuery.name(), version);
        }
        catch (ServiceCallException e) {
            throw new BugTrackerException("Component does not exist: " + componentString, e);
        }
    }

    public @NonNull List<RadarProblem> getProblemData(List<String> problemIds, List<String> customFields) throws JsonSerializationException, IOException {
        Map<String, String> headers = RadarClient.generateHeadersForGetProblem(customFields);
        ArrayList<RadarProblem> retrievedProblems = new ArrayList<RadarProblem>();
        ProblemSyncQueue pendingProblems = new ProblemSyncQueue(problemIds);
        int currentBatch = 1;
        while (!pendingProblems.isEmpty()) {
            LOGGER.debug("Retrieving Problem batch {}. Approximately {} batches remaining", (Object)currentBatch++, (Object)((int)Math.ceil((float)pendingProblems.size() / 15.0f)));
            List<String> batchIds = pendingProblems.getNextBatch();
            try {
                List<RadarProblem> batchProblemData = this.retrieveProblemData(batchIds, headers);
                retrievedProblems.addAll(batchProblemData);
                pendingProblems.updateWithFailedItems(batchIds, batchProblemData);
            }
            catch (ServiceCallException e) {
                LOGGER.warn("Error retrieving problem data. Problem data will be fetched up to {} times per problem due to flakiness in the API.", (Object)3, (Object)e);
                pendingProblems.reAdd(batchIds);
            }
        }
        return retrievedProblems;
    }

    private List<RadarProblem> retrieveProblemData(List<String> batchIds, Map<String, String> headers) throws ServiceCallException, JsonSerializationException, IOException {
        Preconditions.checkArgument((batchIds.size() <= 15 ? 1 : 0) != 0, (String)"Radar API allows for fetching a maximum of %s Problems at a time.", (int)15);
        List<Map<String, Object>> responseJsons = batchIds.size() == 1 ? this.requestSingleProblem(batchIds.getFirst(), headers) : this.requestProblemList(batchIds, headers);
        return this.convertValidProblems(responseJsons);
    }

    private List<Map<String, Object>> requestProblemList(List<String> problemIds, Map<String, String> headers) throws ServiceCallException, IOException, JsonSerializationException {
        String idsString = String.join((CharSequence)",", problemIds);
        try (ResponseBody response = (ResponseBody)Retrofit.executeServiceCall(this.radar.getProblems(idsString, headers)).orElseThrow();){
            List list = (List)JsonUtils.deserializeFromJson((String)response.string(), (TypeReference)new TypeReference<List<Map<String, Object>>>(this){
                {
                    Objects.requireNonNull(this$0);
                }
            });
            return list;
        }
    }

    private List<Map<String, Object>> requestSingleProblem(String problemId, Map<String, String> headers) throws ServiceCallException, IOException, JsonSerializationException {
        try (ResponseBody response = (ResponseBody)Retrofit.executeServiceCall(this.radar.getProblems(problemId, headers)).orElseThrow();){
            List<Map<String, Object>> list = Collections.singletonList((Map)JsonUtils.deserializeFromJson((String)response.string(), (TypeReference)new TypeReference<Map<String, Object>>(this){
                {
                    Objects.requireNonNull(this$0);
                }
            }));
            return list;
        }
    }

    private List<RadarProblem> convertValidProblems(List<Map<String, Object>> responseObjects) {
        List<Map> nonErrors = responseObjects.stream().filter(obj -> {
            if (!obj.containsKey(STATUS_RESPONSE_FIELD)) {
                return true;
            }
            return !this.errorStatusPattern.matcher(obj.get(STATUS_RESPONSE_FIELD).toString()).matches();
        }).toList();
        return (List)JsonUtils.getObjectMapper().convertValue(nonErrors, (TypeReference)new TypeReference<List<RadarProblem>>(this){
            {
                Objects.requireNonNull(this$0);
            }
        });
    }

    private List<Long> findBundleGroupIds(List<String> componentBundleGroupNames) throws BugTrackerException {
        ArrayList<Long> result = new ArrayList<Long>();
        for (String componentBundleGroupName : componentBundleGroupNames) {
            List<RadarComponentBundleGroup> componentBundleGroups = this.findComponentBundleGroups(componentBundleGroupName);
            RadarComponentBundleGroup matchingComponentBundleGroup = (RadarComponentBundleGroup)CollectionUtils.filter(componentBundleGroups, componentBundleGroup -> componentBundleGroup.name().equals(componentBundleGroupName)).stream().findFirst().orElseThrow();
            result.add(matchingComponentBundleGroup.id());
        }
        return result;
    }

    public static class RadarAuthenticator {
        private final A3Authenticator a3authenticator;

        public RadarAuthenticator(A3Authenticator a3authenticator) {
            this.a3authenticator = a3authenticator;
        }

        public RadarAuthenticationToken requestAuthenticationToken() throws ServiceCallException {
            IRadarApi radar = (IRadarApi)Retrofit.builder((String)this.a3authenticator.getRadarBaseUrl()).withNoAuthentication().create(IRadarApi.class);
            try {
                String appleA3Token = this.a3authenticator.requestA3Token();
                return radar.requestAuthenticationToken(this.a3authenticator.getTeamscaleAppleIdmsAppId(), appleA3Token, this.a3authenticator.getA3IdmsPersonId());
            }
            catch (StorageException e) {
                throw new ServiceCallException("Could not retrieve A3 token", (Throwable)e);
            }
        }
    }

    private static class ProblemSyncQueue {
        private final Deque<String> idQueue;
        private final CounterSet<String> failedAttempts = new CounterSet();

        private ProblemSyncQueue(List<String> ids) {
            this.idQueue = new ArrayDeque<String>(ids);
        }

        private List<String> getNextBatch() {
            Preconditions.checkState((!this.idQueue.isEmpty() ? 1 : 0) != 0, (Object)"There are no more IDs left in the queue");
            return Stream.generate(this.idQueue::poll).takeWhile(Objects::nonNull).limit(15L).toList();
        }

        private void reAdd(List<String> ids) {
            for (String id : ids) {
                int count = this.failedAttempts.inc((Object)id);
                if (count >= 3) {
                    LOGGER.error("Will not attempt to sync problem {} more than {} times", (Object)id, (Object)3);
                    continue;
                }
                this.idQueue.add(id);
            }
        }

        private void updateWithFailedItems(List<String> requestedIds, List<RadarProblem> foundProblems) {
            if (foundProblems.size() < requestedIds.size()) {
                Set successfulIds = CollectionUtils.mapToSet(foundProblems, RadarProblem::getIdString);
                this.reAdd(CollectionUtils.filter(requestedIds, id -> !successfulIds.contains(id)));
            }
        }

        private boolean isEmpty() {
            return this.idQueue.isEmpty();
        }

        private int size() {
            return this.idQueue.size();
        }
    }
}

