Using system variables to control program flow might lead to inconsistent behavior dependent on the user or system environment.
Instead, consider using configuration settings or explicit input parameters to manage control flow.

# What Does This Check Look For?
This check searches for instances where ABAP statements access environment-specific information to control program flow, such as system variables (e.g., `SY-UNAME` or `SY-SYSID`), return values from methods of the `CL_ABAP_CONTEXT_INFO` class, or the XCO SY library (i.e., `XCO_CP=>SY->user()`).

# Why is This a Problem?
Relying on system variables for control flow can introduce fragility, as outcomes may unpredictably vary based on the environment or user session characteristics.
Program logic should be robust to changes in system context, promoting consistent behavior across different sessions.

# How Can I Resolve This?
To address this issue, extract and utilize explicit parameters or configurable settings that dictate program behavior irrespective of user or platform specifics.
This approach enhances predictability and adaptability of the program code.

# Examples

## Non-Compliant

```abap
IF SY-UNAME = 'ADMIN'.
    " non-compliant: program flow depends on user name
    PERFORM special_admin_task.
ENDIF.
```

## Compliant

```abap
DATA user_role TYPE string.

CALL FUNCTION 'GET_USER_ROLE'
    EXPORTING
        user_name = SY-UNAME
    IMPORTING
        role      = user_role.

IF user_role = 'ADMIN_ROLE'.
    " compliant: program flow depends on role, not user name
    PERFORM special_admin_task.
ENDIF.
```

# Where Can I Learn More?
- [ABAP Programming Guidelines - ABAP System Fields](https://help.sap.com/doc/abapdocu_752_index_htm/7.52/en-US/abensystem_fields.htm)
