Creating user-managed threads in Java EE containers disrupts the container's ability to fully manage concurrent processing. Utilize Java EE's managed concurrency utilities instead.

# What Does This Check Look For?

This check identifies code that directly uses threads or provides classes/interfaces based on the `Runnable` interface, which is contrary to Java EE specifications. Specifically, it flags when:
- A class inherits from `Thread`.
- A class implements `Runnable`.
- An interface extends `Runnable`.
- The `start` method is called on an object that inherits from `Thread`.

# Why is This a Problem?

Creating and managing threads manually in a Java EE environment can lead to problems such as memory leaks, performance degradation, and inconsistent behavior because it circumvents the thread management provided by the container. The Java EE platform is designed to manage threading on its own to ensure that applications are scalable, robust, and that their components are lifecycle-managed in a standard manner. Interfering with this process by manually managing threads can cause unexpected issues and make the application harder to debug and maintain.

# How Can I Resolve This?

Consider using the managed concurrency utilities that Java EE provides, such as `@Asynchronous` beans. These are designed to work within the lifecycle managed context of Java EE, ensuring that thread-based activities are scalable and managed consistently according to the Java EE platform's standards.

# Examples

## Non-Compliant

```java
Thread thread = new Thread(); 
thread.start(); // This bypasses Java EE's managed concurrency.

class MyThreadedClass extends Thread { } // Inheritance from Thread is not recommended.

class MyRunnableClass implements Runnable { } // Implementing Runnable should be avoided.

interface MyThreadedInterface extends Runnable { } // Extending Runnable is not advisable.
```

## Compliant

```java
import javax.ejb.Asynchronous;
import javax.ejb.Stateless;

@Stateless
public class MyAsynchronousClass {

    @Asynchronous
    public void asyncMethod() {
        // Method logic here will be executed asynchronously, managed by Java EE.
    }
}
```

# Where Can I Learn More?

- [Oracle Docs: Java EE 8 Specification](https://javaee.github.io/javaee-spec/javadocs/)
- [Java EE 7 Tutorial: Concurrency Utilities](https://docs.oracle.com/javaee/7/tutorial/concurrency-utilities.htm)
