`goto` statements that jump backwards or into another block are very error-prone.
They can often be resolved using a `for` or `while` loop.

# What Does This Check Look For?

This check detects `goto` statements that jump backwards or into another block.

# Why is This a Problem?

`goto` statements that jump backwards or into another block should be avoided in all cases.
They are especially error-prone because they often lead to complex and hard to follow code.
In some cases, they can even create unspecified behavior.

Note that in some situations making restricted use of **forward** `goto` statement can be advised.
One such example is the cleaning up of allocated resources in C code.

# How Can I Resolve This?

Backwards `goto`s can often be resolved by rewriting the code using a `for` or `while` loop.

# Examples
## Non-Compliant

```C
void someFunction () {
    top:
        if (exitCondition) {
            goto done;      // compliant: jumps forward
        }
        // do something
 
        if (condition) {
            goto top;      // non-compliant: backwards jump
        }
        goto inside:       // non-compliant: jumps into a block
 
    for (int i = 0; i < 10; i++) {
    inside:
        // do something else
    }
 
    done:
        // cleanup
}
```

## Compliant

```C
void someFunction () {
    while(condition) {
        if (exitCondition) {
            goto done;      // compliant: jumps forward
        }
        // do something
    }
 
    for (int i = 0; i < 10; i++) {
        // do something else
    }
    
    done:
        // cleanup
}
```

# Where Can I Learn More?

- Guideline: MISRA C:2012: 15.2 - The `goto` statement shall jump to a label declared later in the same function
- Guideline: MISRA C:2012: 15.3 - Any label referenced by a `goto` statement shall be declared in the same block, or in any block enclosing the `goto` statement
- Guideline: AUTOSAR C++14, M6-6-1 - Any label referenced by a goto statement shall be declared in the same block, or in a block enclosing the goto statement 
- Guideline: AUTOSAR C++14, M6-6-2 - The goto statement shall jump to a label declared later in the same function body 
