Objects of class `HttpClient` should not be instantiated within `using` statements.
Use a single instance or `IHttpClientFactory` instead.

## Why is This a Problem?

Even though `HttpClient` implements the `IDisposable` interface, underlying sockets are not released
for up to four minutes when an `HttpClient`is disposed.
The `using` statement is a way to deal with disposable objects, i.e. any object declared and instantiated
within the `using` statement is disposed when leaving the `using` block.
In case of `HttpClient` objects this can lead to _socket exhaustion_ (no available ports left) when multiple connections are created within `using` statements.
Then the program is not able to create any more connections and may crash.

## How Can I Resolve This?

There are basically two options to resolve this problem.

First, you could use just a single instance of `HttpClient` for managing the connections instead of creating a new one for each connection.
This way, the program reuses the opened sockets, which eliminates the risk of socket exhaustion.

Second, you could make use of the `IHttpClientFactory` interface (`.NET Core 2.1` and newer) to configure and manage your `HttpClient` instances in one place.
It separates the `HttpClient` from its built in message handlers that manage the connections, such that reusing of sockets becomes possible even with multiple  `HttpClientInstances`.

*Note*: In comparison to the first option, using the `IHttpClientFactory` allows multiple parallel connections while this is not possible using one single `HttpClient` instance.

### Example (Before)

The following code snippet creates 10 connections to a website.
The `HttpClient` instances are disposed when exiting the `using` block, but the 10 sockets will remain open.
In larger scales, i.e. more connections, this will lead to _socket exhaustion_.
```csharp
public class Program {
    public static async Task Main(string[] args) {
    
        // create 10 connection to foo.bar
        for(int i = 0; i<10; i++) {
            using(var client = new HttpClient()) {
                var result = await client.GetAsync("http://foo.bar");
            }
        }
        // 10 open sockets remain
    }
}   
```

### Example (After)

#### Option 1

Create just one single instance of `HttpClient` that then is reused for all connections.
Now, just one single socket is used.

```csharp
public class Program {

    // create a single instance of HttpClient
    private static HttpClient Client = new HttpClient();
    
    public static async Task Main(string[] args) {
    
        // create 10 connection to foo.bar
        for(int i = 0; i<10; i++) {
            
            // reuse the instance for all connections
            var result = await client.GetAsync("http://foo.bar");
            
        }
        // 1 open socket remains
    }
}   
```

#### Option 2

Inject a single instance of `IHttpClientFactory` that then manages the `HttpClient` instances.
Now, even though single `HttpClient` objects are created for each connection,
the program will not cause socket exhaustion problems, because the message handling (which blocks the sockets) is
now done by the central factory.

```csharp
public class Program {

    // create a single instance of HttpClientFactory
    private IHttpClientFactory _httpClientFactory;
    
    // Inject the factory (see note below)
    public Program(IHttpClientFactory factory) {
        _httpClientFactory = factory;
    }
    public static async Task Main(string[] args) {
    
        // create 10 connection to foo.bar
        for(int i = 0; i<10; i++) {
            // create HttpClient instances
            var client = _httpClientFactory.createClient();
            var result = await client.GetAsync("http://foo.bar");
            
        }
    }
}   
```

*Note*: To inject and use the `IHttpClientFactory`, you need to register it first in your `startup.cs` file, e.g. like:
```csharp
public void ConfigureServices(IServiceCollection services) {
        services.AddHttpClient();
}
```

*Note*: There are multiple ways to use the `IHttpClientFactory`.
For more information, refer to the linked references.

## Where Can I Learn More?

### References

- [Microsoft Docs: Issues with the original HttpClient class available in .NET](https://docs.microsoft.com/en-us/dotnet/architecture/microservices/implement-resilient-applications/use-httpclientfactory-to-implement-resilient-http-requests#issues-with-the-original-httpclient-class-available-in-net)
- [Microsoft Docs: Make HTTP requests using IHttpClientFactory in ASP.NET Core](https://docs.microsoft.com/en-us/aspnet/core/fundamentals/http-requests?view=aspnetcore-6.0)
