Using long strings in your code may unnecessarily increase the size of the compiled binaries. 
Consider using shorter strings or managing string resources externally if applicable.

# What Does This Check Look For?
This check identifies and reports cases where strings exceed a configured character limit within the codebase.

The check can be configured to consider only calls to specific functions or macros and search the parameters for long string arguments.

# Why is This a Problem?
Using excessively long strings can lead to larger binaries that will take up SRAM size.
It may also indicate potential inefficiencies in data handling or missed opportunities for resource optimization.

# How Can I Resolve This?
You might consider refactoring the code to use shorter strings. 
Also, consider moving long strings to external resource files or databases indexed by shorter codes that can replace long strings in the actual source. 

# Examples
## Non-Compliant

```C
char * str[50] = "...this is a very long string above the max length"; 

SYS_LOG("[ERROR-123]: this is a very long error description above the max length...");

```

## Compliant

```C
char * str[40] = "...this is a shorter string below the max length";

SYS_LOG("[ERROR-123]"); // ERROR-123 maps to a longer description in an external resources file.
// or have an enum, for instance:
typedef enum {
    ERROR_NONE,
    ERROR_FILE_NOT_FOUND,
    ERROR_PERMISSION_DENIED,
    ERROR_OUT_OF_MEMORY,
    ERROR_INVALID_INPUT
} ErrorCode;
SYS_LOG(ERROR_FILE_NOT_FOUND); // where each enum item maps to an expanded description of the error externally
```

# Where Can I Learn More?
- [Memory Layout of C Program](https://www.hackerearth.com/practice/notes/memory-layout-of-c-program/)
- [Data segment](https://en.wikipedia.org/wiki/Data_segment)
- [The Dark Side of Error Logging](https://arne-mertz.de/2018/03/the-dark-side-of-error-logging/)