Good method names are usually so good that the RETURNING parameter does not need a 
name of its own. The name would do little more than repeat the method name or some other 
obvious information.

For consistency, `RETURNING` parameters should therefore be called `result`.
Tolerable deviations from this rule are cases in which the result is non-obvious, 
for example in methods that return `me` for method chaining, or in methods that 
create something but don't return the created entity but only its ID.

# What Does This Check Look For?

This check reports findings for all `RETURNING` parameters that are not named `result`.
It can be configured to also accept `result` with a hungarian notation prefix (e.g., `rv_result`).

# Why is This a Problem?

Having arbitrarily named `RETURNING` parameters comes with several drawbacks:
In a properly named method, a special name for the returning parameter would add little value.
If the parameter is called like a class member, this would even force you to add an otherwise 
superfluous `me->` when referring to the member.

Moreover, for someone only looking at the method implementation, the name `result`
immediately signals that this is in fact the returning parameter, even before checking the
method definition. 

# How Can I Resolve This?

In order to make it obvious which variable the `RETURNING` parameter is, 
simply call the parameter `result`, or something like `rv_result` if you prefer Hungarian notation.

# Examples
## Non-Compliant

```ABAP
METHODS get_name
  RETURNING
    VALUE(name) TYPE string. " non-compliant: Parameter is called `name`.

METHOD get_name.
  name = me->name. " access to class member has to be prefixed with `me->`.
ENDMETHOD.
```

## Compliant

```ABAP
METHODS get_name
  RETURNING
    VALUE(result) TYPE string. " compliant: Parameter is called `result`.

METHOD get_name.
  result = name. " access to class member `name` does not need `me->` prefix.
ENDMETHOD.
```

# Where Can I Learn More?

- [Clean ABAP: Consider calling the RETURNING parameter RESULT](https://github.com/SAP/styleguides/blob/main/clean-abap/CleanABAP.md#consider-calling-the-returning-parameter-result)
- [code pal for ABAP: Returning Name](https://github.com/SAP/code-pal-for-abap/blob/master/docs/checks/returning-name.md)
