This check detects the usage of old Date and Calendar API classes, which are not recommended for use in Java SE 8 and later.
# Why is This a Problem?
Classes from old Date and Calendar API (`java.util.Date`, `java.util.Calendar`, `java.text.SimpleDateFormat`, etc.) aren’t thread-safe which leading to potential concurrency issues. 
Some date and time classes also exhibit quite poor API design.
For example, years in `java.util.Date` start at 1900, months start at 1 and days start at 0, which is not very intuitive.

# How Can I Resolve This?
The new Date and Time API has been introduced with Java SE 8 in order to address these problems.
The newly presented APIs are immutable and thread safe against possible concurrency issues.
These new APIs are also ISO-centric and follow consistent domain models for date, time, duration and periods.
It is recommended to migrate/refactor your code to use new Date and Time API introduced in Java SE 8.
Some examples of how to migrate to this new API can be seen below:

## Example (Before)

```java
//Getting current time
Date now = new Date();


//Representing specific time
Date birthDay = new GregorianCalendar(1990, Calendar.DECEMBER, 15).getTime();


// Truncating
Calendar now = Calendar.getInstance();
now.set(Calendar.MINUTE, 0);
now.set(Calendar.SECOND, 0);
now.set(Calendar.MILLISECOND, 0);
Date truncated = now.getTime();


// Time formatting and parsing
SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
Date now = new Date();
String formattedDate = dateFormat.format(now);
Date parsedDate = dateFormat.parse(formattedDate);

// Timezone handling
Date nowUtc = new Date();
TimeZone berlinZone =  TimeZone.getTimeZone("Europe/Berlin");
Calendar nowBerlin = Calendar.getInstance(berlinZone);
nowBerlin.setTime(nowUtc);

```

## Example (After)

```java
//Getting current time
ZonedDateTime now = ZonedDateTime.now();


//Representing specific time
LocalDate birthDay = LocalDate.of(1990, Month.DECEMBER, 15);


// Truncating
LocalTime truncated = LocalTime.now().truncatedTo(ChronoUnit.HOURS);


// Time formatting and parsing
LocalDate now = LocalDate.now();
DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
String formattedDate = now.format(formatter);
LocalDate parsedDate = LocalDate.parse(formattedDate, formatter);

// Timezone handling
Instant now = Instant.now();
ZoneId berlinZone = ZoneId.of("Europe/Berlin");
ZonedDateTime nowBerlin = ZonedDateTime.ofInstant(now, berlinZone);

```


## References
- [Java SE 8 Package Summary](https://docs.oracle.com/javase/8/docs/api/java/time/package-summary.html)
