This check detects concatenation of string literals using `+`, `||` or the `CONCAT` function.

# Why is This a Problem?
Concatenating two or more string literals unnecessarily makes the code more complex and less maintainable.
In addition, the concatenation operation is an unnecessary computing overhead.

When concatenating string literals with non-literals in a PL/SQL statement, the database will process and store a new statement for each change of the non-literal value, i.e. it performs a "hard" parse.
For each of these hard parses, the database checks a statement's syntax, verifies the present table/column names and optimizes the query for the scheduler.
Hence, each change to the non-literals (e.g. by user input) triggers a new hard parse and a new statement is stored in the shared pool.
This is inefficient in both CPU and memory load.

In addition, such literal concatenation could enable SQL injections in your application if not used safely.

# How Can I Resolve This?
Concatenations of two or more string literals should be simplified by directly defining the result (e.g. `'A' + 'B'` becomes `'AB'`) to simplify the code.

In addition, instead of concatenating literals with non-literals in a statement, you should use **bind variables** to use the non-literal value in the resulting statement.
Bind variables enable "soft" parsing, i.e. the SQL statement is already present in the shared pool and the value of the bind variable is used.
This way, the corresponding statement is parsed only once and the bind variables act like placeholders in the statement and are replaced by the (changing) values.
As a result, changes to the bind variable do not cause new parsing of the statement at runtime.

## Example (Before)

```oraclesqlplus
SOME_QUERY := 'SELECT * FROM someTable WHERE id="' + input + '"';
```

## Example (After)

```oraclesqlplus
VARIABLE user_input_bind varchar2(50) := input;
SOME_QUERY := 'SELECT * FROM someTable WHERE id=":user_input_bind"'; -- Compliant: Bind variable acts like placeholder for user input
```

# Where Can I Learn More?
## References
- [IT Tutorial: Hard Parse vs. Soft Parse in Oracle](https://ittutorial.org/hard-parse-vs-soft-parse-and-bind-variable-in-oracle-database/)
- [Oracle-Base: Literals, Substitution Variables and Bind Variables](https://oracle-base.com/articles/misc/literals-substitution-variables-and-bind-variables)
