The `switch` statement contains an unusual construct and is not considered well-formed.
A well-formed `switch` statement shall conform to the following rules:
1. `case` and `default` cases shall be closed by exit statements such as `break` (i.e. no fall-through into next `case` is possible)
2. The `default` case shall either appear as first or last `case`

# Why is This a Problem?

The syntax of `switch` statements is not very restrictive.
This may lead to unstructured and inconsistent code that is hard to read and error-prone.
For example, missing exit statements can lead to unintended fall-throughs with undesirable results.
Additionally, ensuring that a `default` case is either the first or last `case` improves the readability, as it makes it easier for others to find the `default` case.

## Exceptions

The above rules may be disregarded in some cases. The following sections show some examples.

### Intended fall-through

A [fallthrough attribute](https://en.cppreference.com/w/cpp/language/attributes/fallthrough) may be added as the last statement of a `case` clause.

```cpp
void foo(int x) {
    switch (x) {
        case 1:
            do_something_with(x);
            [[fallthrough]];        // intended fall-through
        case 2:
            ++x;
            break;
        default;
            break;
    }
}
```

### Omitting `break` statements in the `default` case

Some coding guidelines do not require to have an exit statement in the `default` case if it is the last `case` of the `switch`.
Usually this would lead to findings as it violates rule 1, but this can be tolerated by turning off the respective check option.

# How Can I Resolve This?

The switch statement should be transformed into a semantically equivalent code that does not violate the rules above.
This includes adding unconditional `break` statements in every `case` and moving the `default` case to the bottom (or top).

## Example (Before)

```cpp
void foo(int x) {
    switch (x) {
        case 1:
            do_something_with(x);   // non-compliant, but intended
        case 2:
            ++x;
            break;
        default:                    // non-compliant position of the default case
            break;
        case 3:
        case 4:
            x = 5;
            break;              
    }
}
```

## Example (After)

```cpp
void foo(int x) {
    switch (x) {
        case 1:
            do_something_with(x);   // same logic, but with break statement
            ++x;                    // achieved by copying the logic from second case
            break;
        case 2:
            ++x;
            break;
        case 3:
        case 4:
            x = 5;
            break;
        default:                    // moved default case
            break;          
    }
}
```

# How Can I Resolve This?

## Guidelines

- MISRA C 2012: Rule 16.1 - All `switch` statements shall be well-formed
- AUTOSAR C++14: Rule M6-4-3 - A `switch` statement shall be a well-formed `switch` statement.

## References

This check does not fully conform to Rule 16.1 from MISRA C 2012.
Some parts of the rule are implemented in their own Teamscale check.
The following checks need to be enabled to achieve the same behavior:
- `Switch` statement without default `case`
- `Switch` clauses have to be directly enclosed by a body of `switch` statement (enabled by default)
- `Switch` statement with single possible execution path (enabled by default)
