Testing exceptions using the `@Test(expected = Exception.class)` annotation can lead to misleading test results if further assertions are placed in the same test method.
Instead, consider using `Assert.assertThrows` for JUnit version >= 4.13 or adjust your strategies for older versions.

# What Does This Check Look For?

This check reports a finding when a test method annotated with `@Test(expected = Exception.class)` contains additional assertions within the same method after the line that throws the exception.
Such assertions will not be executed if the exception is thrown, leading to potentially misleading test results.

# Why is This a Problem?

The problem with placing assertions subsequent to code that throws an exception is that these assertions will never execute once the exception is raised.
This could lead developers to mistakenly believe that their assertions have run and passed, when in fact they have not been executed at all.
**This issue may obscure the true state of the program after the exception occurs.**

# How Can I Resolve This?

To resolve this issue, you should:
- For JUnit >= 4.13, use `org.junit.Assert.assertThrows` to explicitly check for exceptions without relying on the `@Test` annotation’s `expected` parameter.
This allows for additional assertions after the exception logic.
- For JUnit versions < 4.13, you may use the `try-catch` idiom to manually verify the exception and still perform assertions on your code's state thereafter.
This approach does not require you to refactor as much code and is suitable for projects that do not support lambdas.

# Examples
## Non-Compliant

```java
@Test(expected = IndexOutOfBoundsException.class)
public void testShouldFail() {
  get();
  // non-compliant: This assertion won't execute, potentially misleading the developer.
  Assert.assertEquals(0, 1);
}

private Object get() {
  throw new IndexOutOfBoundsException();
}
```

## Compliant

```java
@Test
public void testToString() {
    Object obj = get();
    // compliant: Use assertThrows to handle exception and allow further assertions.
    Assert.assertThrows(IndexOutOfBoundsException.class, () -> obj.toString());
    Assert.assertEquals(0, 1); // This assertion is now executed
}
```

```java
@Test
public void testShouldFail() {
    Object obj = get();
    try {
        obj.toString();
        Assert.fail("Expected an IndexOutOfBoundsException to be thrown");
    } catch (IndexOutOfBoundsException e) {
        // compliant: The exception is handled explicitly here.
    }
    Assert.assertEquals(0, 1); // This assertion can now be validated
}
```

# Where Can I Learn More?

- [JUnit Exception Testing](https://github.com/junit-team/junit4/wiki/Exception-testing)

