Missing (or empty) interface documentation makes it hard to get a quick overview of the interface provided by a class or API.
Add appropriate documentation comments to avoid this.

# What Does This Check Look For?

This check detects interfaces which are not documented with an explanatory comment. 
This also includes cases where comments are present, but don't add any additional information.
In the default configuration the analysis only evaluates public functions, public classes and their public members.

# Why is This a Problem?

Missing interface documentation makes it hard to get a quick overview of the interface provided by a class or API.

# How Can I Resolve This?

Add a comment using the language-appropriate documentation mechanism.
For example, in Java use a Javadoc comment; in Python use docstrings.

# Examples
## Non-Compliant

Java:
```java
class Bicycle {
	public int gears = 21;

	// ride the bike (currently a normal comment)
	public void ride() {
		System.out.println("I want to ride my bicycle!");
	}
	
	/** @param m    <-- Empty comment **/
	public void throwException(String m) {
		throw new Exception(m);
	}
}
```

Python:
```python
class Person:
    # print a greeting (currently a normal comment)
    def greet(self):
        print('Hello')
```

## Compliant

Java:
```java
/** Bicycle fit for a bicycle race */
class Bicycle {
	/** Number of available gears */
	public int gears = 21;

	/** Ride the bike (now a Javadoc comment) */
	public void ride() {
		System.out.println("I want to ride my bicycle!");
	}
	
	/** 
     * Throws a generic exception
     * @param m the exception message 
     */
	public void throwException(String m) {
		throw new Exception(m);
	}
}
```

Python:
```python
class Person:
    """This person can say: 'Hello!'"""
    
    def greet(self):
        """Print a greeting (now uses Docstring)"""
        print('Hello')
```

# Where Can I Learn More?

- Python: [PEP 257: Docstring Conventions](https://www.python.org/dev/peps/pep-0257/)
- C#: [C# Language Specification: Annex D Documentation comments](https://docs.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/documentation-comments)
- C++: [Doxygen: Documenting the code](https://www.doxygen.nl/manual/docblocks.html)
- Kotlin: [Document Kotlin code: KDoc and Dokka](https://kotlinlang.org/docs/kotlin-doc.html)
- Java: [Oracle Technical Article: How to Write Doc Comments for the Javadoc Tool](https://www.oracle.com/technical-resources/articles/java/javadoc-tool.html)
- Javascript: [Wordpress Coding Standards Handbook: JavaScript Documentation Standards](https://developer.wordpress.org/coding-standards/inline-documentation-standards/javascript/)
- Matlab: [MathWorks Help Center: Add Help for Your Program](https://www.mathworks.com/help/matlab/matlab_prog/add-help-for-your-program.html)
