# Description
Finds `IF` constructs that can be changed to `CASE`.

# Extended Information
https://github.com/SAP/styleguides/blob/main/clean-abap/CleanABAP.md#prefer-case-to-else-if-for-multiple-alternative-conditions

If the first comparison is a boolean compare, no issue is reported.

# Examples
**Bad example**
```abap
IF l_fcat-fieldname EQ 'FOO'.
ELSEIF l_fcat-fieldname = 'BAR'
    OR l_fcat-fieldname = 'MOO'.
ENDIF.
```

**Good example**
```abap
CASE l_fcat-fieldname.
  WHEN 'FOO'.
  WHEN 'BAR' OR 'MOO'.
ENDCASE.
```

# Clean ABAP Code Style Guide
```ABAP
CASE type.
  WHEN type-some_type.
    " ...
  WHEN type-some_other_type.
    " ...
  WHEN OTHERS.
    RAISE EXCEPTION NEW /clean/unknown_type_failure( ).
ENDCASE.
```

`CASE` makes it easy to see a set of alternatives that exclude each other.
It can be faster than a series of `IF`s because it can translate to a different microprocessor command
instead of a series of subsequently evaluated conditions.
You can introduce new cases quickly, without having to repeat the discerning variable over and over again.
The statement even prevents some errors that can occur when accidentally nesting the `IF`-`ELSEIF`s.

```ABAP
" anti-pattern
IF type = type-some_type.
  " ...
ELSEIF type = type-some_other_type.
  " ...
ELSE.
  RAISE EXCEPTION NEW /dirty/unknown_type_failure( ).
ENDIF.
```

---
For more information visit the [official ABAPLint rule documentation](https://rules.abaplint.org/change_if_to_case) <br>
The contents within section `Clean ABAP Code Style Guide` were copied from the [official SAP Code Style Guides repository](https://github.com/SAP/styleguides/).
