# Rule 21.16 The pointer arguments to the Standard Library function *memcmp* shall point to either a pointer type, an *essentially signed* type, an *essentially unsigned* type, an *essentially Boolean* type or an *essentially enum* type

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99, C11

## Rationale
The Standard Library function
`int memcmp ( const void *s1, const void *s2, size_t n );`
performs a byte by byte comparison of the first `n` bytes of the two objects pointed at by `s1` and `s2`.

Structures shall not be compared using *memcmp* as it may incorrectly indicate that two structures are not equal, even when their members hold the same values. Structures may contain padding with an indeterminate value between their members and *memcmp* will include this in its comparison. It cannot be assumed that the padding will be equal, even when the values of the structure members are the same. Unions have similar concerns along with the added complication that they may incorrectly be reported as having the same value when the representation of different, overlapping members are coincidentally the same.

Objects with *essentially floating* type shall not be compared with *memcmp* as the same value may be stored using different representations.

If an *essentially char* array contains a null character, it is possible to treat the data as a character string rather than simply an array of characters. However that distinction is a matter of interpretation rather than syntax. Since *essentially char* arrays are most frequently used to store character strings, an attempt to compare such arrays using *memcmp* (rather than *strcmp* or *strncmp*) may indicate an error as the number of characters to be compared will be determined by the value of the `size_t` argument rather than the location of the null characters used to terminate the strings. The result may therefore depend on the comparison of characters which are not part of the respective strings.

## Example
```c
struct S;

/* 
 * Return value may indicate that 's1' and 's2' are different due to padding.
 */
bool_t f1 ( struct S *s1, struct S *s2 )
{
  return ( memcmp ( s1, s2, sizeof ( struct S ) ) != 0 ); /* Non-compliant */
}

union U
{
  uint32_t range;
  uint32_t height;
};

/* 
 * Return value may indicate that 'u1' and 'u2' are the same
 * due to unintentional comparison of 'range' and 'height'.
 */
bool_t f2 ( union U *u1, union U *u2 )
{
  return ( memcmp ( u1, u2, sizeof ( union U ) ) != 0 ); /* Non-compliant */
}

const char a[ 6 ] = "task";

/*
 * Return value may incorrectly indicate strings are different as the 
 * length of 'a' (4) is less than the number of bytes compared (6).
 */
bool_t f3 ( const char b[ 6 ] )
{
  return ( memcmp ( a, b, 6 ) != 0 ); /* Non-compliant */
}
```

## See also
Rule 21.14, Rule 21.15

---

Copyright The MISRA Consortium Limited © [Date - March 2025].