# Dir 4.6 typedefs that indicate size and signedness should be used in place of the basic integer types

## Category
Advisory

## Applies to
C90, C99, C11

## Amplification
The basic integer types are the standard integer types excluding *_Bool* (i.e. the *signed* and *unsigned* forms of *char*, *short*, *int*, *long*, and *long long*

Notes:
1. These Guidelines do not treat “plain” *char* as a numerical type (see Section 5.10.2 on *essentially character* types).
2. It is not necessary to use *typedefs* in place of the floating types as the characteristics of floating point types extend beyond just their size.

For C99 and later, the types provided by `<stdint.h>` should be used. For C90, where the standard header `<stdint.h>` is not available, equivalent types should be defined and used. The following definitions may be suitable, but may need to be adjusted to suit the project’s implementation:

```c
typedef signed   char    int8_t;
typedef signed   short   int16_t;
typedef signed   int     int32_t;
typedef signed   long    int64_t;

typedef unsigned char    uint8_t;
typedef unsigned short   uint16_t;
typedef unsigned int     uint32_t;
typedef unsigned long    uint64_t;
```

## Rationale
In situations where the amount of memory being allocated is important, using specific-length types makes it clear how much storage is being reserved for each object.

Adherence to this guideline does **not** guarantee portability because the size of the *int* type may determine whether or not an expression is subject to integer promotion. For example, an expression with type *int16_t* will not be promoted if *int* is implemented using 16 bits but will be promoted if *int* is implemented using 32 bits. This is discussed in more detail in the section on integer promotion in Appendix C.

Note: defining a specific-length type whose size is **not** the same as the implemented type is counter-productive both in terms of storage requirements and in terms of portability. Care should be taken to avoid defining types with the wrong size.

If abstract types are defined in terms of a specific-length type then it is not necessary, and may even be undesirable, for those abstract types to specify the size or sign. For example, the following code defines an abstract type representing mass in kilograms but does not indicate its size or sign:

`typedef uint16_t mass_kg_t;`

It might be desirable not to apply this guideline when interfacing with the Standard Library or code outside the project’s control.

## Exception
1. The basic numerical types may be used in a *typedef* to define a specific-length type.
2. This Directive does not apply to the *main* function, as *main* relies on the implementation-specific sized result that is passed back to the environment, and on the implementation-specific sized integer argument *argc* where provided.
3. It is not necessary to use *typedefs* in the declaration of bit-fields.

## Example
```c
/* Non-compliant - int used to define an object */
int x = 0;

/* Compliant     - int used to define specific-length type */
typedef int SINT_16;

/* Non-compliant - no sign or size specified */
typedef int speed_t;

/* Compliant     - further abstraction does not need specific length */
typedef int16_t torque_t;
```

These following examples are non-exclusive, and other implementation-defined forms are permitted.

```c
void main( void );                        /* Compliant                */
int  main( void );                        /* Compliant by Exception 2 */
void main( int argc, char *argv[] );      /* Compliant by Exception 2 */
int  main( int argc, char *argv[] );      /* Compliant by Exception 2 */
```

Use of *typedefs* is permitted with the *main* function, but they should be compatible with the standard type *int*.

```c
/* Compliant - provided that int32_t is compatible with int           */
int32_t main( void );
```

---

Copyright The MISRA Consortium Limited © [Date - March 2025].