# Rule 30.0.2 Reads and writes on the same file stream shall be separated by a positioning operation

## Category
Required

## Analysis
Undecidable, System

## Amplification
An explicit, interleaving stream positioning operation shall be used between input operations and
output operations on a `std::basic_filebuf`.

This rule applies to direct and indirect calls (e.g. from `std::fstream`) to `std::basic_filebuf`.

*Note:* for the purposes of this rule, a call to `fflush` after an output operation is considered to be an
explicit file positioning operation.

## Rationale
The C `FILE *` abstraction, used as the underlying system file input/output for `std::basic_filebuf`,
holds a single file position that is used when reading from or writing to the file. Using an input operation
on a `FILE *` immediately after an output operation (or vice versa) results in *undefined behaviour*, unless
an interleaving file positioning operation is used to update the file's position.

In addition, a `streambuf` object keeps separate buffer positions for reading and writing characters
from its internal buffer. A `basic_filebuf` object is only guaranteed to synchronize the separate
internal `streambuf` read and write positions that it maintains when a positioning operation is called
when alternating between reading and writing (and vice versa). Failure to include such a positioning
operation leads to *undefined behaviour*.

The accessible positioning operations for `streambuf` are `pubseekoff` and `pubseekpos`, whilst for
file streams they are `tellg`, `seekg`, `tellp`, and `seekp`. One of these functions shall be called when
switching from output to input, or vice versa.

## Example
```cpp
void show_fstream_non_compliant()
{
 std::fstream f { "hello.txt" };

 f << "Hello world!\n" << std::flush; // flush is not a positioning operation

 std::string s {};
 std::getline( f, s ); // Non-compliant - undefined behaviour
}

void show_fstream_compliant()
{
 std::fstream f { "hello.txt" };

 f << "Hello world!\n";

 std::string s {};
 f.seekg( 0, std::ios_base::beg );
 std::getline(f, s); // Compliant - s holds "Hello world!"
}
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
