# Rule 13.6 The operand of the `sizeof` operator shall not contain any expression which has potential *side effects*

## Category
Mandatory

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99

## Amplification
Any expressions appearing in the operand of a `sizeof` operator are not normally evaluated. This rule
mandates that the evaluation of any such expression shall not contain *side effects*, whether or not it is
actually evaluated.

A function call is considered to be a *side effect* for the purposes of this rule.

## Rationale
The operand of a `sizeof` operator may be either an expression or may specify a type. If the operand
contains an expression, a possible programming error is to expect that expression to be evaluated
when it is actually not evaluated in most circumstances.

The C90 standard states that expressions appearing in the operand are not evaluated at run-time.

In C99, expressions appearing in the operand are usually not evaluated at run-time. However, if the
operand contains a variable-length array type then the array size expression will be evaluated if
necessary. If the result can be determined without evaluating the array size expression then it is
unspecified whether it is evaluated or not.

## Exception
An expression of the form `sizeof ( V )`, where `V` is an *lvalue* with a *volatile* qualified type that is not
a variable-length array, is permitted.

## Example
```c
volatile int32_t i;
 int32_t j;
 size_t s;
s = sizeof ( j ); /* Compliant */
s = sizeof ( j++ ); /* Non-compliant */
s = sizeof ( i ); /* Compliant - exception */
s = sizeof ( int32_t ); /* Compliant */
```
In this example the final `sizeof` expression illustrates how it is possible for a variable-length array size
expression to have no effect on the size of the type. The operand is the type "array of `n` pointers
to function with parameter type array of `v` `int32_t`". Because the operand has variable-length
array type, it is evaluated. However, the size of the array of `n` function pointers is unaffected by the
parameter list for those function pointer types. Therefore, the *volatile*-qualified object `v` may or may
not be evaluated and its *side effects* may or may not occur.
```c
volatile uint32_t v;
void f ( int32_t n )
{
 size_t s;
 s = sizeof ( int32_t[ n ] ); /* Compliant */
 s = sizeof ( int32_t[ n++ ] ); /* Non-compliant */
 s = sizeof ( void ( *[ n ] ) ( int32_t a[ v ] ) ); /* Non-compliant */
}
```

## See also
Rule 18.8

---

Copyright The MISRA Consortium Limited © [Date - March 2013].
