# Rule 23.5 A generic selection should not depend on implicit pointer type conversion

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Applies to
C11

## Amplification
This rule only applies to selection based on pointer types. Selection based on an arithmetic type is
never a violation of this rule.

This rule applies when a default association is selected for a controlling expression whose type could,
in other contexts (such as when passed as an argument to a function), be implicitly converted to
another type explicitly listed in the association list.

## Rationale
The controlling expression of a generic selection undergoes value conversion before having its type
compared to any of the entries in the association list, but this is the only conversion applied to its
type. It can then only match an association exactly, or select the default association.

No attempt is made to implicitly convert the type to match an association.

For instance, listing an association for `void *` will never match a pointer to a complete object type,
and listing an association for `T const *` will never match `T *`. This is different from function
arguments, and may therefore cause surprise for users of a generic function implemented as a
generic association, if a type that would be implicitly convertible to an explicitly-listed type is instead
allowed to fall through to the default association.

## Example
Given:
```c
void handle_pi ( int32_t *); /* No associations have this signature */
void handle_cpi (const int32_t *);
void handle_any (void *);
```

Using the following generic function with a pointer to `const int32_t` is compliant, but using a
pointer to (mutable) `int32_t` is not, because there is no implicit conversion to qualified-pointer in
generic matching:
```c
#define handle_pointer1(X) ( _Generic ((X) \
 , const int32_t *: handle_cpi \
 , default : handle_any) (X) )

const int32_t ci;
handle_pointer1 (&ci); /* Compliant - const int32_t * is selected */

int32_t mi;
handle_pointer1 (&mi); /* Non-compliant - default is selected
 NOT const int32_t * */
```

Using the following generic function with a (mutable) pointer to `int32_t` is non-compliant, as there
is no conversion to qualified pointer or to pointer to `void` in generic matching.
```c
#define handle_pointer2(X) ( _Generic ((X) \
 , void *: handle_any \
 , const int32_t *: handle_cpi \
 , default : handle_any) (X))

handle_pointer2 (&mi); /* Non-compliant - default is selected
 NOT void * or const int32_t * */
```

A generic function that wishes to handle pointers to any object type and distinguish them by
qualification, can wrap the controlling expression such that the pointed-to type is explicitly converted
to a pointer to `void`:
```c
void handle_unq (void *);
void handle_cq (void const *);
void handle_vq (void volatile *);
void handle_cvq (void const volatile *);

#define handle_pointer(X) ( _Generic (1 ? (X) : (void *)(X) \
 , void * : handle_unq \
 , void const * : handle_cq \
 , void volatile * : handle_vq \
 , void const volatile * : handle_cvq) (X) )
/* Uses are always compliant */
```
Per the specification of the conditional operator's composite result type, any argument to the macro
with a pointer to object type will be converted to a similarly-qualified pointer to `void` before it is used
as the controlling expression.

## See also
Rule 23.3, Rule 23.4, Rule 23.6

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
